<?php
// rifa.php — Vista principal de Rifas (listado, alta y edición)
session_start();
date_default_timezone_set('America/Guayaquil');
include(__DIR__ . "/conex.php");

$usuario   = $_SESSION['usuario']   ?? '';
$idusuario = (int)($_SESSION['idusuario'] ?? 0);
$sucursal  = $_SESSION['sucursal']  ?? '';

if (!$idusuario) {
  header("Location: ../administracion.php");
  exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>Sistema de Gestión · Rifas</title>

  <!-- Favicon inline (evita 404 a crud.ico) -->
  <link rel="icon"
        href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>🎟️</text></svg>">

  <!-- CSS -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css">

  <style>
    :root { --sidebar-w: 250px; --sidebar-w-collapsed: 80px; }
    body { background:#f4f6f9; padding-top:60px; overflow-x:hidden; }

    /* Topbar */
    .topbar{
      width:100%; height:60px; background:#343a40; color:#fff;
      display:flex; align-items:center; justify-content:space-between;
      padding:0 16px; position:fixed; top:0; left:0; z-index:1000;
    }
    .toggle-btn{ cursor:pointer; }

    /* Sidebar contenedor (menu.php debe renderizar #sidebar) */
    .sidebar{
      width:var(--sidebar-w); background:#343a40; padding:16px; height:100vh;
      position:fixed; left:0; top:60px; display:flex; flex-direction:column; align-items:flex-start;
      transition:width .25s ease; z-index:1000; overflow-x:hidden; overflow-y:auto;
    }
    .sidebar.collapsed{ width:var(--sidebar-w-collapsed); }
    .sidebar a{ color:#fff; text-decoration:none; padding:10px 12px; display:flex; align-items:center; width:100%;
      transition:background .2s, transform .1s; border-radius:8px; font-size:14px; }
    .sidebar a:hover, .sidebar a.active{ background:#495057; transform:scale(1.02); }

    /* Contenido */
    .main-content{ margin-left:var(--sidebar-w); padding:1rem; transition:margin-left .25s ease; }
    .sidebar.collapsed ~ #mainContent{ margin-left:var(--sidebar-w-collapsed); }

    /* Switch */
    .switch{ position:relative; display:inline-block; width:52px; height:28px; }
    .switch input{ opacity:0; width:0; height:0; }
    .slider{ position:absolute; cursor:pointer; inset:0; background:#c9c9c9; transition:.2s; border-radius:28px; }
    .slider:before{ content:""; position:absolute; height:22px; width:22px; left:3px; bottom:3px; background:#fff; transition:.2s; border-radius:50%; }
    input:checked + .slider{ background:#4CAF50; }
    input:checked + .slider:before{ transform:translateX(24px); }

    /* Tablas */
    table.dataTable thead th { font-size:13px; }
    table.dataTable tbody td { font-size:12px; vertical-align:middle; }

    /* Modales: z-index correcto */
    .modal-backdrop { z-index:1040 !important; }
    .modal-dialog   { z-index:1050 !important; }

    /* Botones / inputs compactos */
    .btn, .form-control, .form-select{ font-size:14px; }
  </style>
</head>
<body>

  <!-- Topbar -->
  <div class="topbar">
    <div class="d-flex align-items-center gap-2">
      <div class="toggle-btn" onclick="toggleSidebar()" aria-label="Alternar menú"><i class="fas fa-bars"></i></div>
      <span class="fw-semibold">Sistema de Gestión · Rifas</span>
    </div>
    <div class="info d-flex gap-3 align-items-center">
      <span><i class="fa-regular fa-user"></i> <?php echo htmlspecialchars($usuario); ?></span>
      <span id="fechaActual"></span>
      <span id="horaActual"></span>
      <button class="btn btn-sm btn-outline-light" onclick="cerrar()"><i class="fa-solid fa-right-from-bracket"></i> Salir</button>
    </div>
  </div>

  <!-- Sidebar -->
  <?php
  if (file_exists(__DIR__."/menu.php")) {
    include(__DIR__."/menu.php"); // Debe pintar un <aside id="sidebar" class="sidebar">...</aside>
  } else {
    // Menú mínimo opcional
    echo '<aside id="sidebar" class="sidebar">
            <a href="#" class="active"><i class="fa-solid fa-ticket me-2"></i>Rifas</a>
          </aside>';
  }
  ?>

  <!-- Contenido Principal -->
  <div class="main-content" id="mainContent">
    <div class="container-fluid">
      <div class="card shadow-sm">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-center mb-3">
            <h5 class="mb-0"><i class="fa-solid fa-ticket me-2"></i>Rifas Programadas</h5>
            <button type="button" class="btn btn-success" data-bs-toggle="modal" data-bs-target="#agregar">
              <i class="fa-solid fa-plus"></i> Agregar Rifa
            </button>
          </div>

          <div class="table-responsive">
            <table id="usuar" class="table table-hover table-striped w-100">
              <thead>
                <tr class="table-info">
                  <th>Lotería</th>
                  <th>Fecha</th>
                  <th>Día</th>
                  <th>Hora Inicio</th>
                  <th>Hora Fin</th>
                  <th>Activo</th>
                  <th>Acción</th>
                </tr>
              </thead>
              <tbody><!-- DataTables por AJAX --></tbody>
            </table>
          </div>

        </div>
      </div>
    </div>
  </div>

  <!-- Modal Agregar -->
  <div class="modal fade" id="agregar" tabindex="-1" aria-labelledby="agregarLabel" aria-hidden="true">
    <div class="modal-dialog">
      <div class="modal-content">
        <div class="modal-header py-2">
          <h6 class="modal-title" id="agregarLabel">Agregar nueva rifa</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
        </div>

        <div class="modal-body">
          <form id="formAgregarUsuario" onsubmit="event.preventDefault(); guardar();">
            <div class="row g-3">
              <div class="col-md-6">
                <label for="fecha" class="form-label">Fecha*</label>
                <input name="fecha" id="fecha" type="date" class="form-control" required>
              </div>
              <div class="col-md-6">
                <label for="dia" class="form-label">Día*</label>
                <input type="text" class="form-control" id="dia" name="dia" required readonly>
              </div>

              <div class="col-12">
                <label for="loteria" class="form-label">Lotería*</label>
                <select class="form-select" id="loteria" name="loteria" required></select>
              </div>

              <div class="col-md-6">
                <label for="hora_inicio" class="form-label">Hora inicio*</label>
                <input type="time" class="form-control" id="hora_inicio" name="hora_inicio" required min="00:00" max="23:59" value="06:00">
              </div>
              <div class="col-md-6">
                <label for="hora_fin" class="form-label">Hora fin*</label>
                <input type="time" class="form-control" id="hora_fin" name="hora_fin" required min="00:00" max="23:59" value="19:55">
              </div>
            </div>
          </form>
        </div>

        <div class="modal-footer py-2">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
          <button type="button" class="btn btn-primary" onclick="guardar()">Guardar</button>
        </div>
      </div>
    </div>
  </div>

  <!-- Modal Editar -->
  <div id="modalEditar" class="modal fade" tabindex="-1" aria-labelledby="modalEditarLabel" aria-hidden="true">
    <div class="modal-dialog">
      <div class="modal-content">
        <div class="modal-header py-2">
          <h6 class="modal-title" id="modalEditarLabel">Editar Rifa</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
        </div>

        <div class="modal-body">
          <form id="formEditar" onsubmit="event.preventDefault(); actualizar();">
            <input type="hidden" id="idloteria_a" name="idloteria_a">
            <div class="row g-3">
              <div class="col-md-6">
                <label for="fecha_a" class="form-label">Fecha*</label>
                <input name="fecha_a" id="fecha_a" type="date" class="form-control" required>
              </div>
              <div class="col-md-6">
                <label for="dia_a" class="form-label">Día*</label>
                <input type="text" class="form-control" id="dia_a" name="dia_a" required readonly>
              </div>

              <div class="col-12">
                <label for="loteria_a" class="form-label">Lotería*</label>
                <select class="form-select" id="loteria_a" name="loteria_a" required></select>
              </div>

              <div class="col-md-6">
                <label for="hora_inicio_a" class="form-label">Hora inicio*</label>
                <input type="time" class="form-control" id="hora_inicio_a" name="hora_inicio_a" required min="00:00" max="23:59">
              </div>
              <div class="col-md-6">
                <label for="hora_fin_a" class="form-label">Hora fin*</label>
                <input type="time" class="form-control" id="hora_fin_a" name="hora_fin_a" required min="00:00" max="23:59">
              </div>
            </div>
          </form>
        </div>

        <div class="modal-footer py-2">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
          <button type="button" class="btn btn-primary" onclick="actualizar()">Guardar Cambios</button>
        </div>
      </div>
    </div>
  </div>

  <!-- JS (sin duplicados) -->
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  <script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
  <script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <script>
  // ====== Reloj topbar ======
  (function reloj(){
    function actualizarHora() {
      const ahora = new Date();
      const opcionesFecha = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      const opcionesHora  = { hour: '2-digit', minute: '2-digit', second: '2-digit' };
      document.getElementById('fechaActual').textContent = ahora.toLocaleDateString('es-ES', opcionesFecha);
      document.getElementById('horaActual').textContent  = ahora.toLocaleTimeString('es-ES', opcionesHora);
    }
    setInterval(actualizarHora, 1000); actualizarHora();
  })();

  // ====== Sidebar toggle ======
  function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    const mainContent = document.getElementById('mainContent');
    if (!sidebar) return;
    sidebar.classList.toggle('collapsed');
    mainContent.style.marginLeft = sidebar.classList.contains('collapsed')
      ? getComputedStyle(document.documentElement).getPropertyValue('--sidebar-w-collapsed')
      : getComputedStyle(document.documentElement).getPropertyValue('--sidebar-w');
  }

  // ====== DataTable: listado por AJAX (solo vigentes) ======
  let tablaRifa = null;
  $(function () {
    tablaRifa = $('#usuar').DataTable({
      processing: true,
      serverSide: false,
      ajax: {
        url: 'rifa_listado.php',
        type: 'GET',
        data: function () {
          return {
            vigentes: 1,      // rifas de hoy (no terminadas) + futuras
            // activo: 1,     // descomenta si quieres solo activadas
            _: Date.now()     // anti-cache
          };
        },
        dataSrc: (json) => json.data || [],
        error: function (xhr) {
          const msg = xhr.responseText || 'Error Ajax';
          Swal.fire('Error', msg.substring(0, 800), 'error');
        }
      },
      columns: [
        { data: 'loteria_nombre' },
        { data: 'fecha' },
        { data: 'dia' },
        { data: 'hora_inicio' },
        { data: 'hora_fin' },
        {
          data: 'activo',
          className:'text-center',
          render: function (val, type, row) {
            const checked = String(val) === '1' ? 'checked' : '';
            return `
              <label class="switch mb-0">
                <input type="checkbox" class="toggle-switch" data-id="${row.id}" ${checked}>
                <span class="slider"></span>
              </label>`;
          }
        },
        {
          data: null,
          orderable: false,
          className:'text-center',
          render: function (row) {
            return `
              <div class="btn-group btn-group-sm" role="group">
                <button type="button" class="btn btn-primary toggle-editar"
                  data-id="${row.id}"
                  data-loteria="${row.loteria_nombre}"
                  data-fecha="${row.fecha}"
                  data-dia="${row.dia}"
                  data-hora_i="${row.hora_inicio}"
                  data-hora_f="${row.hora_fin}">
                  <i class="fa-solid fa-pen-to-square"></i>
                </button>
                <button type="button" class="btn btn-danger toggle-condicion" data-id="${row.id}">
                  <i class="fa-solid fa-trash"></i>
                </button>
              </div>`;
          }
        }
      ],
      order: [[1,'asc'], [3,'asc']],
      pageLength: 25,
      language: { url: "https://cdn.datatables.net/plug-ins/1.13.8/i18n/es-ES.json" },
      responsive: true
    });
  });
  function refrescarTabla(){ if (tablaRifa) tablaRifa.ajax.reload(null, false); }

  // ====== Activar/Desactivar (switch) — usa el estado actual ======
  $(document).on("change", ".toggle-switch", function () {
    const $sw = $(this);
    const id  = Number($sw.data("id"));
    const nuevoEstado = $sw.is(":checked") ? 1 : 0;

    $.ajax({
      url: "actulizar_estado_r.php", // si renombraste a actualizar_estado_r.php, cámbialo aquí
      type: "POST",
      data: { idloteria: id, estado: nuevoEstado },
      dataType: "json"
    }).done(function (r) {
      if (r && r.success) {
        refrescarTabla();
      } else {
        Swal.fire("Atención", (r && r.message) || "No se pudo cambiar el estado.", "warning");
        $sw.prop("checked", !nuevoEstado);
      }
    }).fail(function () {
      Swal.fire("Error", "Error de conexión con el servidor.", "error");
      $sw.prop("checked", !nuevoEstado);
    });
  });

  // ====== Eliminar (condicion=0) ======
  $(document).on("click", ".toggle-condicion", function () {
    const id = $(this).data("id");
    Swal.fire({
      title: "¿Estás seguro?",
      text: "La rifa será eliminada (borrado lógico).",
      icon: "warning",
      showCancelButton: true,
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
      confirmButtonText: "Sí, eliminar",
      cancelButtonText: "Cancelar"
    }).then((res) => {
      if (res.isConfirmed) {
        $.ajax({
          url: "actualizar_condicion_r.php",
          type: "POST",
          data: { idloteria: id },
          dataType: "json"
        }).done(function (r) {
          if (r && r.success) {
            Swal.fire({ icon: "success", title: "Rifa eliminada", timer: 1800, showConfirmButton:false });
            refrescarTabla();
          } else {
            Swal.fire("Error", "No se pudo eliminar.", "error");
          }
        }).fail(function(){
          Swal.fire("Error", "Error de conexión con el servidor.", "error");
        });
      }
    });
  });

  // ====== Guardar (crear) ======
  function guardar() {
    const fecha       = $('#fecha').val().trim();
    const dia         = $('#dia').val().trim();
    const loteria     = $('#loteria').val();
    const hora_inicio = $('#hora_inicio').val();
    const hora_fin    = $('#hora_fin').val();

    if (!fecha || !dia || !loteria || !hora_inicio || !hora_fin) {
      Swal.fire("Error", "Completa todos los campos obligatorios.", "error"); return;
    }
    if (hora_inicio >= hora_fin) {
      Swal.fire("Error", "La hora de inicio debe ser menor que la de fin.", "error"); return;
    }

    const $btn = $('#agregar .btn.btn-primary').prop('disabled', true).text('Guardando...');
    $.ajax({
      url: 'guardar_loteria.php',
      type: 'POST',
      data: { fecha, dia, loteria, hora_inicio, hora_fin },
      dataType: 'json'
    }).done(function (r) {
      if (r && r.success) {
        $('#agregar').modal('hide');
        Swal.fire({ title:'Éxito', text:'Rifa guardada correctamente.', icon:'success', timer:1500, showConfirmButton:false });
        $('#formAgregarUsuario')[0].reset();
        setMinYHoy(document.getElementById('fecha')); // restablece min + día
        refrescarTabla();
      } else {
        Swal.fire("Atención", (r && r.message) ? r.message : "No se pudo guardar.", "warning");
      }
    }).fail(function (xhr) {
      const msg = xhr.responseJSON?.message || xhr.responseText || 'Error al guardar';
      Swal.fire("Error", msg, "error");
    }).always(function(){
      $btn.prop('disabled', false).text('Guardar');
    });
  }

  // ====== Cargar datos al modal de edición ======
  $(document).on('click', '.toggle-editar', function() {
    const id   = $(this).data('id');
    const lot  = $(this).data('loteria');
    const fec  = $(this).data('fecha');
    const dia  = $(this).data('dia');
    const hi   = $(this).data('hora_i');
    const hf   = $(this).data('hora_f');

    $('#idloteria_a').val(id);
    $('#fecha_a').val(fec).trigger('change');
    $('#dia_a').val(dia);
    $('#hora_inicio_a').val(hi);
    $('#hora_fin_a').val(hf);

    const selEdit = document.getElementById('loteria_a');
    if (selEdit && ![...selEdit.options].some(o => o.value === lot)) {
      selEdit.appendChild(new Option(lot, lot));
    }
    $('#loteria_a').val(lot).trigger('change');

    new bootstrap.Modal('#modalEditar').show();
  });

  // ====== Actualizar (editar) ======
  function actualizar() {
    const id          = $('#idloteria_a').val();
    const loteria     = $('#loteria_a').val();
    const fecha       = $('#fecha_a').val();
    const dia         = $('#dia_a').val();
    const hora_inicio = $('#hora_inicio_a').val();
    const hora_fin    = $('#hora_fin_a').val();

    if (!id || !loteria || !fecha || !dia || !hora_inicio || !hora_fin) {
      Swal.fire("Error", "Completa todos los campos obligatorios.", "error"); return;
    }
    if (hora_inicio >= hora_fin) {
      Swal.fire("Error", "La hora de inicio debe ser menor que la de fin.", "error"); return;
    }

    const $btn = $('#modalEditar .btn.btn-primary').prop('disabled', true).text('Guardando...');
    $.ajax({
      url: 'guardar_edicion_l.php',
      type: 'POST',
      data: { idloteria_a: id, loteria_a: loteria, fecha_a: fecha, dia_a: dia, hora_inicio_a: hora_inicio, hora_fin_a: hora_fin },
      dataType: 'json'
    }).done(function (r) {
      if (r && r.success) {
        $('#modalEditar').modal('hide');
        Swal.fire({ title:'Éxito', text:'Datos actualizados correctamente.', icon:'success', timer:1500, showConfirmButton:false });
        refrescarTabla();
      } else {
        Swal.fire("Atención", (r && r.message) ? r.message : "No se pudo actualizar.", "warning");
      }
    }).fail(function () {
      Swal.fire("Error", "Ocurrió un error al guardar los datos.", "error");
    }).always(function(){
      $btn.prop('disabled', false).text('Guardar Cambios');
    });
  }

  // ====== Llenado de selects + día por fecha + AUTOCOMPLETADO de horas ======
  (function initLoterias() {
    const tz = "America/Guayaquil";
    const diasSemana = ["Domingo","Lunes","Martes","Miércoles","Jueves","Viernes","Sábado"];

    // Ecuador por día (opciones visibles)
    const loteriasPorDiaEC = {
      0: ["CON LTRA DOM"], // Domingo
      1: ["CON LTRA LUN"], // Lunes
      2: ["CON LTT MAR"],  // Martes
      3: ["CON LTRA MIE"], // Miércoles
      4: ["CON LTT JUE"],  // Jueves
      5: ["CON LTRA VIE"], // Viernes
      6: ["CON LTT SAB"]   // Sábado
    };

    // Argentina visibles
    const opcionesAR = [
      { value: "CON MATUTINA",   text: "CON MATUTINA" },
      { value: "CON VESPERTINA", text: "CON VESPERTINA" }
    ];

    // Mapa de horas por tipo
    const HORAS_POR_TIPO = {
      "CON MATUTINA":   { inicio: "06:00", fin: "12:30" },
      "CON VESPERTINA": { inicio: "06:00", fin: "21:00" }
    };
    // ECU: todas 06:00–19:55
    [
      "CON LTRA DOM","CON LTRA LUN","CON LTT MAR",
      "CON LTRA MIE","CON LTT JUE","CON LTRA VIE","CON LTT SAB"
    ].forEach(k => { HORAS_POR_TIPO[k] = { inicio: "06:00", fin: "19:55" }; });

    function horasPorLoteria(valor) {
      return HORAS_POR_TIPO[valor] || null;
    }
    function aplicarHorasSiCorresponde(selectEl, inicioEl, finEl) {
      if (!selectEl || !inicioEl || !finEl) return;
      const cfg = horasPorLoteria(selectEl.value);
      if (!cfg) return;
      if (!inicioEl.value || inicioEl.dataset.autofill !== "0") {
        inicioEl.value = cfg.inicio;
        inicioEl.dataset.autofill = "1";
      }
      if (!finEl.value || finEl.dataset.autofill !== "0") {
        finEl.value = cfg.fin;
        finEl.dataset.autofill = "1";
      }
    }
    function bloquearAutofillAlEditar(input) {
      if (!input) return;
      input.addEventListener("input", () => { input.dataset.autofill = "0"; });
    }

    function hoyYYYYMMDD() {
      const now = new Date(new Date().toLocaleString("en-US", { timeZone: tz }));
      const y = now.getFullYear();
      const m = String(now.getMonth()+1).padStart(2,"0");
      const d = String(now.getDate()).padStart(2,"0");
      return `${y}-${m}-${d}`;
    }
    window.setMinYHoy = function(input) {
      if (!input) return null;
      const hoy = hoyYYYYMMDD();
      input.min = hoy;
      if (!input.value || input.value < hoy) input.value = hoy;
      return input.value;
    }
    function dayIndexFrom(yyyyMMdd) {
      const d = new Date(`${yyyyMMdd}T00:00:00`);
      return isNaN(d) ? null : d.getDay(); // 0..6
    }
    function buildSelect(selectEl, dayIndex) {
      if (!selectEl) return;
      selectEl.innerHTML = "";

      // ECU por día
      const ec = loteriasPorDiaEC[dayIndex] || [];
      ec.forEach(txt => selectEl.appendChild(new Option(txt, txt)));

      // Argentina optgroup
      const group = document.createElement("optgroup");
      group.label = "Argentina";
      opcionesAR.forEach(o => {
        const opt = document.createElement("option");
        opt.value = o.value; opt.text = o.text;
        group.appendChild(opt);
      });
      selectEl.appendChild(group);

      // Si no hay ECU ese día, selecciona la primera de Argentina
      if (ec.length === 0 && selectEl.options.length > 0) {
        for (let i=0; i<selectEl.options.length; i++) {
          const opt = selectEl.options[i];
          if (opt.parentElement && opt.parentElement.label === "Argentina") {
            selectEl.selectedIndex = i; break;
          }
        }
      }
    }

    // Alta
    const dateAlta = document.getElementById("fecha");
    const diaAlta  = document.getElementById("dia");
    const selAlta  = document.getElementById("loteria");
    const hIniAlta = document.getElementById("hora_inicio");
    const hFinAlta = document.getElementById("hora_fin");

    if (dateAlta && diaAlta && selAlta) {
      const v = setMinYHoy(dateAlta);
      const di = dayIndexFrom(v);
      if (di !== null) {
        diaAlta.value = diasSemana[di];
        buildSelect(selAlta, di);
      }
      dateAlta.addEventListener("change", () => {
        const vi = dateAlta.value;
        const di2 = dayIndexFrom(vi);
        if (di2 !== null) {
          diaAlta.value = diasSemana[di2];
          buildSelect(selAlta, di2);
          aplicarHorasSiCorresponde(selAlta, hIniAlta, hFinAlta);
        }
      });

      if (hIniAlta && hFinAlta) {
        bloquearAutofillAlEditar(hIniAlta);
        bloquearAutofillAlEditar(hFinAlta);
        aplicarHorasSiCorresponde(selAlta, hIniAlta, hFinAlta);
        selAlta.addEventListener("change", () => aplicarHorasSiCorresponde(selAlta, hIniAlta, hFinAlta));
      }
    }

    // Edición
    const dateEdit = document.getElementById("fecha_a");
    const diaEdit  = document.getElementById("dia_a");
    const selEdit  = document.getElementById("loteria_a");
    const hIniEdit = document.getElementById("hora_inicio_a");
    const hFinEdit = document.getElementById("hora_fin_a");

    if (dateEdit && diaEdit && selEdit) {
      const v = setMinYHoy(dateEdit);
      const di = dayIndexFrom(v);
      if (di !== null) {
        diaEdit.value = diasSemana[di];
        buildSelect(selEdit, di);
      }
      dateEdit.addEventListener("change", () => {
        const vi = dateEdit.value;
        const di2 = dayIndexFrom(vi);
        if (di2 !== null) {
          diaEdit.value = diasSemana[di2];
          buildSelect(selEdit, di2);
          aplicarHorasSiCorresponde(selEdit, hIniEdit, hFinEdit);
        }
      });

      if (hIniEdit && hFinEdit) {
        bloquearAutofillAlEditar(hIniEdit);
        bloquearAutofillAlEditar(hFinEdit);
        aplicarHorasSiCorresponde(selEdit, hIniEdit, hFinEdit);
        selEdit.addEventListener("change", () => aplicarHorasSiCorresponde(selEdit, hIniEdit, hFinEdit));

        // Re-aplicar al abrir el modal
        document.addEventListener("shown.bs.modal", function (ev) {
          if (ev.target && ev.target.id === "modalEditar") {
            aplicarHorasSiCorresponde(selEdit, hIniEdit, hFinEdit);
          }
        });

        // Asegurar selección correcta con valor ya guardado
        $(document).on('click', '.toggle-editar', function() {
          const val = $(this).data('loteria');
          if (!selEdit) return;
          if ([...selEdit.options].every(o => o.value !== val)) {
            selEdit.appendChild(new Option(val, val));
          }
          $(selEdit).val(val);
          aplicarHorasSiCorresponde(selEdit, hIniEdit, hFinEdit);
        });
      }
    }
  })();

  // ====== Logout ======
  function cerrar() {
    Swal.fire({
      title: '¿Estás seguro?',
      text: '¿Deseas cerrar sesión?',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Sí, cerrar sesión',
      cancelButtonText: 'Cancelar'
    }).then((result) => {
      if (result.isConfirmed) {
        fetch('../../controllers/usuario.php', {
          method: 'POST',
          headers: { 'Content-Type':'application/x-www-form-urlencoded' },
          body: new URLSearchParams({ boton:'cerrar' })
        })
        .then(res => res.ok ? location.href = '../administracion.php' : Promise.reject(res.statusText))
        .catch(err => Swal.fire('Error','Hubo un error al cerrar sesión: '+err,'error'));
      }
    });
  }
  </script>

</body>
</html>
