<?php
// reporte_diario.php — Reporte diario de tbr_ticket a CSV y PDF (20:00 América/Guayaquil)
// - Guarda CSV y PDF en reports/YYYY/MM/
// - Soporta ?fecha=YYYY-MM-DD para rehacer reportes pasados
// - Respuesta JSON con paths generados
//
// Requisitos: FPDF (fpdf.php) y conex.php con $conn (mysqli)

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

$isCli = (PHP_SAPI === 'cli');
if (!$isCli) {
  header('Content-Type: application/json; charset=utf-8');
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
}

require_once __DIR__ . '/../conex.php'; // ajusta la ruta si es necesario
if (!($conn instanceof mysqli)) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error: conexión BD no disponible']); exit;
}
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($conn, 'utf8mb4');

// ==== Parámetros ====
// Permite rehacer reportes pasados con ?fecha=YYYY-MM-DD (por defecto hoy)
$hoy = date('Y-m-d');
$fecha = isset($_GET['fecha']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $_GET['fecha']) ? $_GET['fecha'] : $hoy;

// ==== Consulta ====
// Mantengo tu lógica original con LIKE (compatible si fecha_impreso es DATETIME o VARCHAR).
// Si fecha_impreso es DATETIME y hay índice, para mejor performance usa: WHERE DATE(fecha_impreso)=?
$sql = "SELECT * 
        FROM `ganamass_dbninooso`.`tbr_ticket`
        WHERE `fecha_impreso` LIKE CONCAT('%', ?, '%')
          AND condicion = 1
          AND estado >= 2";

try {
  $stmt = $conn->prepare($sql);
  $stmt->bind_param('s', $fecha);
  $stmt->execute();
  $rs = $stmt->get_result();

  $rows = [];
  while ($row = $rs->fetch_assoc()) {
    $rows[] = $row;
  }

  // ==== Paths de salida ====
  $baseDir = __DIR__ . '/reports/' . date('Y', strtotime($fecha)) . '/' . date('m', strtotime($fecha));
  if (!is_dir($baseDir)) {
    if (!mkdir($baseDir, 0775, true) && !is_dir($baseDir)) {
      throw new RuntimeException('No se pudo crear el directorio de reportes: ' . $baseDir);
    }
  }

  $slugFecha = $fecha; // YYYY-MM-DD
  $csvPath = $baseDir . "/reporte_tickets_{$slugFecha}.csv";
  $pdfPath = $baseDir . "/reporte_tickets_{$slugFecha}.pdf";

  // ==== Generar CSV ====
  $csvOk = generarCSV($csvPath, $rows);

  // ==== Generar PDF (FPDF) ====
  $pdfOk = generarPDF($pdfPath, $rows, $fecha);

  $out = [
    'success' => true,
    'fecha'   => $fecha,
    'total'   => count($rows),
    'csv'     => $csvOk ? rutaPublica($csvPath) : null,
    'pdf'     => $pdfOk ? rutaPublica($pdfPath) : null
  ];

  if ($isCli) {
    // En CLI, imprime un resumen corto para logs
    echo "[OK] {$fecha} total=" . count($rows) . " CSV=" . ($csvOk ? 'SI' : 'NO') . " PDF=" . ($pdfOk ? 'SI' : 'NO') . PHP_EOL;
  } else {
    echo json_encode($out, JSON_UNESCAPED_UNICODE);
  }

} catch (Throwable $e) {
  if ($isCli) {
    fwrite(STDERR, "[ERROR] " . $e->getMessage() . PHP_EOL);
  } else {
    http_response_code(500);
    echo json_encode(['success'=>false, 'message'=>$e->getMessage()]);
  }
}

/* ====================== Helpers ====================== */

/**
 * Genera un CSV con BOM UTF-8 y encabezados dinámicos.
 * @param string $path
 * @param array<int,array<string,mixed>> $rows
 * @return bool
 */
function generarCSV(string $path, array $rows): bool {
  $f = @fopen($path, 'wb');
  if (!$f) return false;

  // Añadir BOM UTF-8 para abrir bien en Excel
  fwrite($f, "\xEF\xBB\xBF");

  if (empty($rows)) {
    // Escribe encabezado mínimo si no hay filas
    fputcsv($f, ['mensaje','detalle'], ';');
    fputcsv($f, ['SIN DATOS','No existen registros para la fecha consultada'], ';');
    fclose($f);
    return true;
  }

  // Encabezados dinámicos
  $headers = array_keys($rows[0]);
  fputcsv($f, $headers, ';');

  foreach ($rows as $r) {
    // Asegura orden de columnas igual al encabezado
    $line = [];
    foreach ($headers as $h) {
      $val = $r[$h] ?? '';
      // Aplana arrays/objetos si existieran (no debería)
      if (is_array($val) || is_object($val)) $val = json_encode($val, JSON_UNESCAPED_UNICODE);
      $line[] = (string)$val;
    }
    fputcsv($f, $line, ';');
  }
  fclose($f);
  return true;
}

/**
 * Genera un PDF simple con tabla usando FPDF.
 * @param string $path
 * @param array<int,array<string,mixed>> $rows
 * @param string $fecha
 * @return bool
 */
function generarPDF(string $path, array $rows, string $fecha): bool {
  // Requiere FPDF
  $fpdfPath = __DIR__ . '/fpdf/fpdf.php';
  if (!file_exists($fpdfPath)) {
    // Intenta ruta alternativa (ajusta si usas otra)
    $fpdfPath = __DIR__ . '/../pdf/fpdf/fpdf.php';
  }
  if (!file_exists($fpdfPath)) {
    // Si no está, generamos un TXT de cortesía para no fallar
    file_put_contents($path . '.faltante.txt', "Falta FPDF (fpdf.php). Instálala y vuelve a ejecutar.");
    return false;
  }
  require_once $fpdfPath;

  // Definir columnas principales (si no existen, se omiten gracefully)
  $cols = [
    'idticket'       => 'ID',
    'fecha_impreso'  => 'Fecha impreso',
    'numero'         => 'Número',
    'valor'          => 'Valor',
    'estado'         => 'Estado',
    'condicion'      => 'Cond.'
  ];
  // Si la tabla usa otros nombres, se ajusta dinámicamente:
  if (!empty($rows)) {
    $keys = array_keys($rows[0]);
    // asegura que solo imprimimos columnas que existan
    $cols = array_filter($cols, fn($k) => in_array($k, $keys, true), ARRAY_FILTER_USE_KEY);
    if (empty($cols)) {
      // Si ninguna clave coincide, toma las primeras 6 columnas cualesquiera
      $dyn = [];
      foreach (array_slice($keys, 0, 6) as $k) {
        $dyn[$k] = ucfirst(str_replace('_', ' ', $k));
      }
      $cols = $dyn;
    }
  }

  $pdf = new FPDF('L', 'mm', 'A4');
  $pdf->AddPage();

  // Encabezado
  $pdf->SetFont('Arial', 'B', 14);
  $pdf->Cell(0, 8, utf8_decode("Reporte diario de tickets — $fecha"), 0, 1, 'C');
  $pdf->SetFont('Arial', '', 10);
  $pdf->Cell(0, 6, utf8_decode("Generado: " . date('Y-m-d H:i:s')), 0, 1, 'C');
  $pdf->Ln(2);

  // Tabla
  $pdf->SetFont('Arial', 'B', 9);
  $anchoCols = calcularAnchos(array_values($cols));
  foreach ($cols as $label) {
    $pdf->Cell($anchoCols[$label], 8, utf8_decode($label), 1, 0, 'C');
  }
  $pdf->Ln();

  $pdf->SetFont('Arial', '', 9);
  foreach ($rows as $r) {
    foreach ($cols as $k => $label) {
      $txt = isset($r[$k]) ? (string)$r[$k] : '';
      $pdf->Cell($anchoCols[$label], 7, utf8_decode($txt), 1, 0, 'L');
    }
    $pdf->Ln();
  }

  // Pie con total
  $pdf->Ln(2);
  $pdf->SetFont('Arial', 'B', 10);
  $pdf->Cell(0, 6, utf8_decode('Total registros: ' . count($rows)), 0, 1, 'R');

  // Guardar
  return (bool)$pdf->Output('F', $path);
}

/**
 * Calcula anchos aproximados para columnas simples.
 * @param array<int,string> $labels
 * @return array<string,float> label->ancho
 */
function calcularAnchos(array $labels): array {
  // Suma total ~ 270mm (A4 horizontal con márgenes)
  $base = count($labels) ?: 1;
  $total = 270.0;
  $min = 30.0; // no menos de 30mm por columna
  $anchos = [];
  $dflt = max($min, $total / $base);
  foreach ($labels as $lb) {
    $anchos[$lb] = $dflt;
  }
  return $anchos;
}

/**
 * Devuelve ruta "pública" relativa para mostrar en JSON.
 * Si sirves este script bajo webroot, puedes adaptar a URL absoluta.
 */
function rutaPublica(string $absPath): string {
  // Intenta generar una ruta relativa desde el directorio actual
  $docroot = realpath($_SERVER['DOCUMENT_ROOT'] ?? __DIR__);
  $abs     = realpath($absPath);
  if ($docroot && $abs && str_starts_with($abs, $docroot)) {
    return ltrim(str_replace('\\', '/', substr($abs, strlen($docroot))), '/');
  }
  // Si no se puede, devuelve absoluta (para CLI/logs)
  return $absPath;
}
