<?php
// views/pages/raspadita/api/venta_confirmar.php
// ✅ Confirma venta final del carrito
// ✅ Pasa reservas a definitivas: estado=3, tmp_cart=0
// ✅ Guarda cabecera/detalle en tablas tbr_venta_raspadita y tbr_venta_raspadita_detalle
// ✅ Valor por cada raspadita desde tbr_raspadita.valor (NO fijo)
declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $j = json_decode($raw, true);
  return is_array($j) ? $j : [];
}

if (!isset($_SESSION['idusuario'])) jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);
$id_usuario = (int)$_SESSION['idusuario'];

$_SESSION['rasp_cart'] = $_SESSION['rasp_cart'] ?? [];
$items = array_values($_SESSION['rasp_cart']);
if (!$items) jexit(['success'=>false,'message'=>'El carrito está vacío.'], 409);

$in = read_json();
$obs = trim((string)($in['observacion'] ?? ''));
if ($obs !== '') $obs = mb_substr($obs, 0, 200);

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);
mysqli_set_charset($db, 'utf8mb4');

// ✅ IMPORTANTE: hora Ecuador para NOW() en MySQL
@mysqli_query($db, "SET time_zone = '-05:00'");

try {
  $db->begin_transaction();

  // ✅ Obtener series únicas del carrito
  $series = [];
  foreach ($items as $it) {
    $s = trim((string)($it['serie'] ?? ''));
    if ($s !== '') $series[$s] = true;
  }
  $series = array_keys($series);
  if (!$series) { $db->rollback(); jexit(['success'=>false,'message'=>'Carrito inválido (sin códigos).'], 400); }

  // ✅ Validar que TODOS sigan reservados tmp_cart=1 (estado=2), pertenecen al usuario y leer valor real
  $sqlVal = "SELECT codigo, COALESCE(valor,0) AS valor
             FROM tbr_raspadita
             WHERE codigo = ?
               AND id_usuario = ?
               AND anulado = 0
               AND estado = 2
               AND tmp_cart = 1
             LIMIT 1";
  $stVal = $db->prepare($sqlVal);
  if (!$stVal) throw new RuntimeException("Prepare validar: ".$db->error);

  $valores = [];     // codigo => valor
  $total = 0.0;

  foreach ($series as $codigo) {
    $stVal->bind_param('si', $codigo, $id_usuario);
    $stVal->execute();
    $res = $stVal->get_result();
    $row = $res ? $res->fetch_assoc() : null;

    if (!$row) {
      $stVal->close();
      $db->rollback();
      jexit(['success'=>false,'message'=>"No se puede confirmar: el código {$codigo} ya no está reservado (tmp_cart=1) o cambió de estado."], 409);
    }

    $valor = (float)($row['valor'] ?? 0);
    if ($valor <= 0) {
      $stVal->close();
      $db->rollback();
      jexit(['success'=>false,'message'=>"El código {$codigo} no tiene valor válido en tbr_raspadita.valor"], 422);
    }

    $valores[$codigo] = $valor;
    $total += $valor;
  }
  $stVal->close();

  // ✅ Insert cabecera venta
  $cantidad = count($series);
  $total = round($total, 2);

  $sqlV = "INSERT INTO tbr_venta_raspadita (id_usuario, fecha, total, cantidad, observacion)
           VALUES (?, NOW(), ?, ?, ?)";
  $stV = $db->prepare($sqlV);
  if (!$stV) throw new RuntimeException("Prepare venta: ".$db->error);

  // id_usuario (i), total (d), cantidad (i), obs (s)
  $stV->bind_param('idis', $id_usuario, $total, $cantidad, $obs);
  $stV->execute();
  $id_venta = (int)$db->insert_id;
  $stV->close();

  if ($id_venta <= 0) { $db->rollback(); jexit(['success'=>false,'message'=>'No se pudo crear la venta.'], 500); }

  // ✅ Insert detalle con valor real por código
  $sqlD = "INSERT INTO tbr_venta_raspadita_detalle (id_venta, codigo, valor)
           VALUES (?, ?, ?)";
  $stD = $db->prepare($sqlD);
  if (!$stD) throw new RuntimeException("Prepare detalle: ".$db->error);

  foreach ($series as $codigo) {
    $valor = (float)$valores[$codigo];
    $stD->bind_param('isd', $id_venta, $codigo, $valor);
    $stD->execute();
  }
  $stD->close();

  // ✅ Confirmar en tbr_raspadita: pasa de reservado a definitivo (estado=3, tmp_cart=0)
  $sqlC = "UPDATE tbr_raspadita
           SET estado = 3, tmp_cart = 0, tmp_cart_at = NULL
           WHERE codigo = ?
             AND id_usuario = ?
             AND anulado = 0
             AND estado = 2
             AND tmp_cart = 1
           LIMIT 1";
  $stC = $db->prepare($sqlC);
  if (!$stC) throw new RuntimeException("Prepare confirmar: ".$db->error);

  $confirmados = 0;
  foreach ($series as $codigo) {
    $stC->bind_param('si', $codigo, $id_usuario);
    $stC->execute();
    if ($stC->affected_rows === 1) $confirmados++;
  }
  $stC->close();

  if ($confirmados !== $cantidad) {
    $db->rollback();
    jexit(['success'=>false,'message'=>'No se pudieron confirmar todos los códigos.'], 409);
  }

  // ✅ Vaciar carrito
  $_SESSION['rasp_cart'] = [];

  $db->commit();

  jexit([
    'success'=>true,
    'message'=>'✅ Venta confirmada correctamente.',
    'id_venta'=>$id_venta,
    'cantidad'=>$cantidad,
    'total'=>$total
  ], 200);

} catch (Throwable $e) {
  try { $db->rollback(); } catch (Throwable $x) {}
  jexit(['success'=>false,'message'=>'Error interno','detail'=>$e->getMessage()], 500);
}