<?php
// views/pages/raspadita/api/venta_cart_add.php
// ✅ Agrega al carrito (sesión)
// ✅ Reserva en BD: estado=2, tmp_cart=1
// ✅ Valor desde tbr_raspadita.valor
declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $j = json_decode($raw, true);
  return is_array($j) ? $j : [];
}

if (!isset($_SESSION['idusuario'])) jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);
$id_usuario = (int)$_SESSION['idusuario'];

$in = read_json();
$serie = trim((string)($in['serie'] ?? ''));
$serie = preg_replace('/\s+/', '', $serie);

if ($serie === '') jexit(['success'=>false,'message'=>'Serie requerida'], 422);

// carrito
$_SESSION['rasp_cart'] = $_SESSION['rasp_cart'] ?? [];
if (!is_array($_SESSION['rasp_cart'])) $_SESSION['rasp_cart'] = [];

// evitar duplicado en sesión
foreach ($_SESSION['rasp_cart'] as $it) {
  if (($it['serie'] ?? '') === $serie) {
    jexit(['success'=>true,'message'=>'Ya estaba en el carrito'], 200);
  }
}

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

try {
  $db->begin_transaction();

  // 1) validar disponible + traer valor
  // Disponible = estado=1 y tmp_cart=0 (ajusta si tu estado "disponible" es otro)
  $sqlGet = "SELECT id, codigo, COALESCE(valor,0) AS valor
             FROM tbr_raspadita
             WHERE codigo = ?
               AND id_usuario = ?
               AND anulado = 0
               AND estado = 2
               AND tmp_cart = 0
             LIMIT 1";
  $st = $db->prepare($sqlGet);
  if (!$st) throw new RuntimeException("Prepare select: ".$db->error);

  $st->bind_param('si', $serie, $id_usuario);
  $st->execute();
  $res = $st->get_result();
  $row = $res ? $res->fetch_assoc() : null;
  $st->close();

  if (!$row) {
    $db->rollback();
    jexit(['success'=>false,'message'=>'No disponible: no pertenece a tu usuario, está anulado o ya está reservado/vendido'], 409);
  }

  $idRasp = (int)$row['id'];
  $valor  = (float)($row['valor'] ?? 0);
  if ($valor <= 0) {
    $db->rollback();
    jexit(['success'=>false,'message'=>'Este código no tiene valor válido en tbr_raspadita.valor'], 422);
  }

  // 2) reservar en BD: estado=2, tmp_cart=1
  $sqlUpd = "UPDATE tbr_raspadita
             SET estado = 2,
                 tmp_cart = 1,
                 tmp_cart_at = NOW()
             WHERE id = ?
               AND codigo = ?
               AND id_usuario = ?
               AND anulado = 0
               AND estado = 2
               AND tmp_cart = 0
             LIMIT 1";
  $stU = $db->prepare($sqlUpd);
  if (!$stU) throw new RuntimeException("Prepare update: ".$db->error);

  $stU->bind_param('isi', $idRasp, $serie, $id_usuario);
  $stU->execute();
  $ok = ($stU->affected_rows === 1);
  $stU->close();

  if (!$ok) {
    $db->rollback();
    jexit(['success'=>false,'message'=>'No se pudo reservar: el código cambió de estado'], 409);
  }

  // 3) guardar en sesión
  $_SESSION['rasp_cart'][] = [
    'id'    => $idRasp,
    'fecha' => date('Y-m-d H:i:s'),
    'serie' => $serie,
    'valor' => $valor
  ];

  $db->commit();

  // total actualizado
  $total = 0.0;
  foreach ($_SESSION['rasp_cart'] as $it) $total += (float)($it['valor'] ?? 0);

  jexit([
    'success'=>true,
    'message'=>'Agregado al carrito',
    'item'=>['id'=>$idRasp,'serie'=>$serie,'valor'=>$valor],
    'total'=>round($total,2)
  ], 200);

} catch (Throwable $e) {
  try { $db->rollback(); } catch (Throwable $x) {}
  jexit(['success'=>false,'message'=>'Error interno','detail'=>$e->getMessage()], 500);
}