<?php
// views/pages/raspadita/api/reporte_coordinador.php
// POST JSON: { "fecha_inicio":"YYYY-MM-DD", "fecha_fin":"YYYY-MM-DD" }
// ✅ id_coordinador = sesión (id_coordinador o idusuario)
// ✅ Entregadas: estado=2 (cant y valor)
// ✅ Vendidas: estado IN (3,8) (cant y valor)
// ✅ Premios total: suma premio en vendidas (3,8)
// ✅ Premios pagados: suma premio estado=8
// ✅ Neto: vendidas_valor - premios_pagados
// ✅ Detalle: lista del rango (por defecto hoy)
// ✅ Ignora anulado=1 (y NO elimina NULL)

declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? '')) !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c = 200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE);
  exit;
}

// ✅ Valida sesión como en admin_coordinador.php
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  jexit(['success'=>false,'message'=>'No autorizado (ingreso)'], 401);
}
if ((int)($_SESSION['id_rol'] ?? 0) !== 7) {
  jexit(['success'=>false,'message'=>'No autorizado (rol)'], 401);
}

// ✅ ID coordinador desde sesión (soporta varias llaves)
$idCoord = (int)($_SESSION['id_coordinador'] ?? 0);
if ($idCoord <= 0) {
  $idCoord = (int)($_SESSION['idusuario'] ?? $_SESSION['id_usuario'] ?? $_SESSION['id'] ?? 0);
}
if ($idCoord <= 0) {
  jexit(['success'=>false,'message'=>'No autorizado (id coordinador)'], 401);
}

$raw = file_get_contents('php://input') ?: '';
$in = json_decode($raw, true);
if (!is_array($in)) { $in = []; }

$hoy = date('Y-m-d');
$fi = trim((string)($in['fecha_inicio'] ?? $hoy));
$ff = trim((string)($in['fecha_fin'] ?? $hoy));

$re = '/^\d{4}-\d{2}-\d{2}$/';
if (!preg_match($re, $fi)) $fi = $hoy;
if (!preg_match($re, $ff)) $ff = $hoy;
if ($fi > $ff) { $tmp = $fi; $fi = $ff; $ff = $tmp; }

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

// Rango [fi 00:00:00, ff 23:59:59]
$iniDT = $fi . " 00:00:00";
$finDT = $ff . " 23:59:59";

// ---- RESUMEN ----
$sqlResumen = "
SELECT
  COALESCE(SUM(CASE WHEN r.estado = 2 THEN 1 ELSE 0 END),0) AS entregadas_cant,
  COALESCE(SUM(CASE WHEN r.estado = 2 THEN r.valor ELSE 0 END),0) AS entregadas_valor,

  COALESCE(SUM(CASE WHEN r.estado IN (3,8) THEN 1 ELSE 0 END),0) AS vendidas_cant,
  COALESCE(SUM(CASE WHEN r.estado IN (3,8) THEN r.valor ELSE 0 END),0) AS vendidas_valor,

  COALESCE(SUM(CASE WHEN r.estado IN (3,8) THEN COALESCE(r.premio,0) ELSE 0 END),0) AS premios_total,
  COALESCE(SUM(CASE WHEN r.estado = 8 THEN COALESCE(r.premio,0) ELSE 0 END),0) AS premios_pagados
FROM tbr_raspadita r
WHERE r.id_coordinador = ?
  AND (r.anulado IS NULL OR r.anulado <> 1)
  AND r.fecha_hora BETWEEN ? AND ?
";

$stmt = mysqli_prepare($db, $sqlResumen);
if (!$stmt) jexit(['success'=>false,'message'=>'Error SQL resumen: '.mysqli_error($db)], 500);

mysqli_stmt_bind_param($stmt, "iss", $idCoord, $iniDT, $finDT);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);
$row = $res ? mysqli_fetch_assoc($res) : [];
mysqli_stmt_close($stmt);

$entCant = (int)($row['entregadas_cant'] ?? 0);
$entVal  = (float)($row['entregadas_valor'] ?? 0);

$venCant = (int)($row['vendidas_cant'] ?? 0);
$venVal  = (float)($row['vendidas_valor'] ?? 0);

$premTot = (float)($row['premios_total'] ?? 0);
$premPag = (float)($row['premios_pagados'] ?? 0);

$neto = $venVal - $premPag;

// ---- DETALLE VENDIDAS ----
$sqlVendidas = "
SELECT r.codigo, r.valor, COALESCE(r.premio,0) AS premio, r.estado, r.fecha_hora
FROM tbr_raspadita r
WHERE r.id_coordinador = ?
  AND (r.anulado IS NULL OR r.anulado <> 1)
  AND r.estado IN (3,8)
  AND r.fecha_hora BETWEEN ? AND ?
ORDER BY r.fecha_hora DESC, r.id DESC
";

$stmt = mysqli_prepare($db, $sqlVendidas);
if (!$stmt) jexit(['success'=>false,'message'=>'Error SQL vendidas: '.mysqli_error($db)], 500);

mysqli_stmt_bind_param($stmt, "iss", $idCoord, $iniDT, $finDT);
mysqli_stmt_execute($stmt);
$rVend = mysqli_stmt_get_result($stmt);

$vendidas = [];
if ($rVend) {
  while ($x = mysqli_fetch_assoc($rVend)) { $vendidas[] = $x; }
}
mysqli_stmt_close($stmt);

// ---- DETALLE PAGADAS (estado=8) ----
$sqlPagadas = "
SELECT r.codigo, r.valor, COALESCE(r.premio,0) AS premio, r.estado, r.fecha_hora
FROM tbr_raspadita r
WHERE r.id_coordinador = ?
  AND (r.anulado IS NULL OR r.anulado <> 1)
  AND r.estado = 8
  AND r.fecha_hora BETWEEN ? AND ?
ORDER BY r.fecha_hora DESC, r.id DESC
";

$stmt = mysqli_prepare($db, $sqlPagadas);
if (!$stmt) jexit(['success'=>false,'message'=>'Error SQL pagadas: '.mysqli_error($db)], 500);

mysqli_stmt_bind_param($stmt, "iss", $idCoord, $iniDT, $finDT);
mysqli_stmt_execute($stmt);
$rPag = mysqli_stmt_get_result($stmt);

$pagadas = [];
if ($rPag) {
  while ($x = mysqli_fetch_assoc($rPag)) { $pagadas[] = $x; }
}
mysqli_stmt_close($stmt);

jexit([
  'success' => true,
  'rango' => ['fecha_inicio'=>$fi, 'fecha_fin'=>$ff],
  'resumen' => [
    'entregadas_cant' => $entCant,
    'entregadas_valor' => round($entVal, 2),
    'vendidas_cant' => $venCant,
    'vendidas_valor' => round($venVal, 2),
    'premios_total' => round($premTot, 2),
    'premios_pagados' => round($premPag, 2),
    'neto' => round($neto, 2),
  ],
  'detalle' => [
    'vendidas' => $vendidas,
    'pagadas'  => $pagadas,
  ],
], 200);