<?php
declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '[]', true);
  return is_array($j) ? $j : [];
}
function table_has_column(mysqli $db, string $table, string $col): bool {
  $sql = "SHOW COLUMNS FROM `$table` LIKE '" . mysqli_real_escape_string($db, $col) . "'";
  $res = mysqli_query($db, $sql);
  return ($res && mysqli_num_rows($res) > 0);
}

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$in = read_json();
$codigos = $in['codigos'] ?? [];
$id_admin = (int)($_SESSION['idusuario'] ?? ($in['id_admin'] ?? 0));
$ip = $_SERVER['REMOTE_ADDR'] ?? '';

if (!is_array($codigos) || count($codigos) === 0) jexit(['success'=>false,'message'=>'codigos requeridos'], 422);

// limpiar codigos
$codigos = array_values(array_unique(array_filter(array_map(static function($c){
  $c = trim((string)$c);
  $c = preg_replace('/\s+/', '', $c);
  return $c !== '' ? $c : null;
}, $codigos))));
if (count($codigos) === 0) jexit(['success'=>false,'message'=>'codigos inválidos'], 422);

$DESCUENTO = 0.50;

// ✅ AJUSTA si tu campo se llama "serie"
$FIELD_CODIGO = 'codigo';

// ✅ detectar columnas saldo (para descontar)
$usuarioTieneSaldo = table_has_column($db, 'tbr_usuario', 'saldo');
$coordTieneSaldo   = table_has_column($db, 'tbr_coordinador', 'saldo'); // <-- ajusta si tu tabla se llama distinto

try{
  mysqli_begin_transaction($db);

  // armar IN ('A','B','C') seguro
  $safe = [];
  foreach ($codigos as $c) {
    $safe[] = "'" . mysqli_real_escape_string($db, $c) . "'";
  }
  $inList = implode(',', $safe);

  // 1) SELECT con FOR UPDATE
  $sqlSel = "
    SELECT $FIELD_CODIGO AS codigo, estado, id_usuario, id_canillita, id_coordinador
    FROM tbr_raspadita
    WHERE $FIELD_CODIGO IN ($inList)
    FOR UPDATE
  ";
  $rs = mysqli_query($db, $sqlSel);
  $rows = [];
  while($r = mysqli_fetch_assoc($rs)) $rows[] = $r;

  if(!count($rows)){
    mysqli_commit($db);
    jexit(['success'=>false,'message'=>'No se encontraron raspaditas con esos códigos'], 404);
  }

  // 2) contar descuentos
  $countPunto = []; // estado=2
  $countCan   = []; // estado=3
  $countCoord = []; // id_coordinador>0
  $skipped = [];

  foreach($rows as $r){
    $estado = (int)($r['estado'] ?? 0);
    $idu = (int)($r['id_usuario'] ?? 0);
    $idc = (int)($r['id_canillita'] ?? 0);
    $idcoord = (int)($r['id_coordinador'] ?? 0);

    // Punto
    if($estado === 2 && $idu > 0){
      $countPunto[$idu] = ($countPunto[$idu] ?? 0) + 1;
    }

    // Canillita
    if($estado === 3 && $idc > 0){
      $countCan[$idc] = ($countCan[$idc] ?? 0) + 1;
    }

    // Coordinador (sin borrar id)
    if($idcoord > 0){
      $countCoord[$idcoord] = ($countCoord[$idcoord] ?? 0) + 1;
    }

    // si no aplica a nada (solo para informar)
    if(!(($estado===2 && $idu>0) || ($estado===3 && $idc>0) || ($idcoord>0))){
      $skipped[] = $r['codigo'];
    }
  }

  // 3) UPDATE masivo ✅ SIN tocar ids
  $sqlUp = "
    UPDATE tbr_raspadita
    SET
      -- baja estado solo si >=4
      estado = CASE WHEN estado >= 4 THEN estado - 1 ELSE estado END,

      -- si estado=2 (punto): suma contador
      devuelto_punto = CASE
        WHEN estado = 2 THEN COALESCE(devuelto_punto,0) + 1
        ELSE COALESCE(devuelto_punto,0)
      END,

      -- si estado=3 (canillita): suma contador
      devuelto_canillita = CASE
        WHEN estado = 3 THEN COALESCE(devuelto_canillita,0) + 1
        ELSE COALESCE(devuelto_canillita,0)
      END,

      -- coordinador (si tiene id)
      devuelto_coordinador = CASE
        WHEN COALESCE(id_coordinador,0) > 0 THEN COALESCE(devuelto_coordinador,0) + 1
        ELSE COALESCE(devuelto_coordinador,0)
      END

    WHERE $FIELD_CODIGO IN ($inList)
  ";
  mysqli_query($db, $sqlUp);
  $updated = mysqli_affected_rows($db);

  // 4) descontar saldos
  $total_descuento = 0.0;

  // Canillita
  foreach($countCan as $id => $cnt){
    $des = $DESCUENTO * $cnt;
    $total_descuento += $des;
    mysqli_query($db, "UPDATE tbr_canillita SET saldo = COALESCE(saldo,0) - ".(float)$des." WHERE id=".(int)$id." LIMIT 1");
  }

  // Punto
  if($usuarioTieneSaldo){
    foreach($countPunto as $id => $cnt){
      $des = $DESCUENTO * $cnt;
      $total_descuento += $des;
      mysqli_query($db, "UPDATE tbr_usuario SET saldo = COALESCE(saldo,0) - ".(float)$des." WHERE idusuario=".(int)$id." LIMIT 1");
    }
  }

  // Coordinador
  if($coordTieneSaldo){
    foreach($countCoord as $id => $cnt){
      $des = $DESCUENTO * $cnt;
      $total_descuento += $des;
      mysqli_query($db, "UPDATE tbr_coordinador SET saldo = COALESCE(saldo,0) - ".(float)$des." WHERE id=".(int)$id." LIMIT 1");
    }
  }

  mysqli_commit($db);

  jexit([
    'success' => true,
    'message' => 'Devolución procesada correctamente (SIN borrar IDs)',
    'updated' => max(0, $updated),
    'total_descuento' => round($total_descuento, 2),
    'skipped' => $skipped,
    'nota_saldo_punto' => !$usuarioTieneSaldo ? 'tbr_usuario no tiene columna saldo; no se descontó saldo del punto' : null,
    'nota_saldo_coord' => !$coordTieneSaldo ? 'tbr_coordinador no tiene columna saldo o la tabla es distinta; no se descontó saldo coordinador' : null
  ]);

}catch(Throwable $e){
  @mysqli_rollback($db);
  jexit(['success'=>false,'message'=>'Fallo al devolver','detail'=>$e->getMessage()], 500);
}
