<?php
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-CSRF-Token');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión BD'], JSON_UNESCAPED_UNICODE);
  exit;
}
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

/* ================= HELPERS ================= */
function jexit(array $p, int $c=200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }

function read_json(): array {
  $r = file_get_contents('php://input') ?: '';
  $j = json_decode($r, true);
  return is_array($j) ? $j : [];
}

function bind(mysqli_stmt $s, string $t, array &$p): void {
  $b = []; $b[] = $t;
  foreach ($p as $k => &$v) { $b[] = &$v; }
  call_user_func_array([$s, 'bind_param'], $b);
}

function numf($v, $d=0.0){
  $n = (float)($v ?? $d);
  return is_finite($n) ? $n : $d;
}

/* ================= INPUT ================= */
$in = read_json();

$fecha_ini = trim((string)($in['fecha_ini'] ?? date('Y-m-d')));
$fecha_fin = trim((string)($in['fecha_fin'] ?? date('Y-m-d')));
$id_punto  = (int)($in['id_punto'] ?? 0);
$id_can    = (int)($in['id_canillita'] ?? 0);

if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha_ini) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha_fin)) {
  jexit(['success'=>false,'message'=>'Formato fecha inválido. Use YYYY-MM-DD'], 422);
}
if ($fecha_ini > $fecha_fin) { $tmp=$fecha_ini; $fecha_ini=$fecha_fin; $fecha_fin=$tmp; }

/* ================= BASE WHERE ================= */
$where = "
  DATE(r.fecha_hora) BETWEEN ? AND ?
  AND COALESCE(r.devuelto_punto,0)=0
  AND COALESCE(r.devuelto_canillita,0)=0
  AND COALESCE(r.devuelto_coordinador,0)=0
  AND COALESCE(r.anulado,0) NOT IN (1,2)
";
$params = [$fecha_ini, $fecha_fin];
$types  = "ss";

if ($id_punto > 0) { $where .= " AND r.id_usuario=?";   $params[]=$id_punto; $types.="i"; }
if ($id_can   > 0) { $where .= " AND r.id_canillita=?"; $params[]=$id_can;   $types.="i"; }

/* ================= TOTALES ================= */
$sqlTot = "
SELECT
  SUM(CASE WHEN r.estado=1 THEN 1 ELSE 0 END) asignadas,
  SUM(CASE WHEN r.estado=1 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_asignado,

  SUM(CASE WHEN r.estado=2 THEN 1 ELSE 0 END) entregadas,
  SUM(CASE WHEN r.estado=2 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_entregado,

  SUM(CASE WHEN r.estado IN (3,8) THEN 1 ELSE 0 END) vendidas,
  SUM(CASE WHEN r.estado IN (3,8) THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_vendido,

  SUM(CASE WHEN r.estado=8 THEN COALESCE(r.premio,0) ELSE 0 END) premios_pagados
FROM tbr_raspadita r
WHERE $where
";
$stmt = mysqli_prepare($db, $sqlTot);
if (!$stmt) jexit(['success'=>false,'message'=>'Error prepare totales: '.mysqli_error($db)], 500);

bind($stmt, $types, $params);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);
$t = mysqli_fetch_assoc($res) ?: [];
mysqli_stmt_close($stmt);

$asignadas       = (int)($t['asignadas'] ?? 0);
$valor_asignado  = numf($t['valor_asignado'] ?? 0);
$entregadas      = (int)($t['entregadas'] ?? 0);
$valor_entregado = numf($t['valor_entregado'] ?? 0);
$vendidas        = (int)($t['vendidas'] ?? 0);
$valor_vendido   = numf($t['valor_vendido'] ?? 0);
$premios_pagados = numf($t['premios_pagados'] ?? 0);
$saldo           = $valor_vendido - $premios_pagados;

/* ==========================================================
   RESUMEN: SOLO PV Y CAN (NO coordinadores)
   (incluye valor_entregado por si lo quieres mostrar)
   ========================================================== */
$sqlRes = "
SELECT
  CASE
    WHEN COALESCE(r.id_canillita,0) > 0 AND COALESCE(c.canillita,'')<>'' THEN CONCAT('CAN: ', c.canillita)
    WHEN COALESCE(r.id_usuario,0)   > 0 AND COALESCE(u.usuario,'')<>''   THEN CONCAT('PV: ', u.usuario)
  END responsable,

  SUM(CASE WHEN r.estado=1 THEN 1 ELSE 0 END) asignadas,
  SUM(CASE WHEN r.estado=1 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_asignado,

  SUM(CASE WHEN r.estado=2 THEN 1 ELSE 0 END) entregadas,
  SUM(CASE WHEN r.estado=2 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_entregado,

  SUM(CASE WHEN r.estado IN (3,8) THEN 1 ELSE 0 END) vendidas,
  SUM(CASE WHEN r.estado IN (3,8) THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_vendido,

  SUM(CASE WHEN r.estado=8 THEN COALESCE(r.premio,0) ELSE 0 END) premios_pagados
FROM tbr_raspadita r
LEFT JOIN tbr_usuario u   ON u.idusuario=r.id_usuario
LEFT JOIN tbr_canillita c ON c.id=r.id_canillita
WHERE $where
  AND (COALESCE(r.id_usuario,0) > 0 OR COALESCE(r.id_canillita,0) > 0)
GROUP BY responsable
ORDER BY responsable ASC
LIMIT 2000
";
$stmt = mysqli_prepare($db, $sqlRes);
if (!$stmt) jexit(['success'=>false,'message'=>'Error prepare resumen: '.mysqli_error($db)], 500);

bind($stmt, $types, $params);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);

$resumen = [];
while($row = mysqli_fetch_assoc($res)){
  $vv = numf($row['valor_vendido'] ?? 0);
  $pp = numf($row['premios_pagados'] ?? 0);

  $resumen[] = [
    'responsable'      => (string)($row['responsable'] ?? ''),
    'asignadas'        => (int)($row['asignadas'] ?? 0),
    'valor_asignado'   => numf($row['valor_asignado'] ?? 0),
    'entregadas'       => (int)($row['entregadas'] ?? 0),
    'valor_entregado'  => numf($row['valor_entregado'] ?? 0),
    'vendidas'         => (int)($row['vendidas'] ?? 0),
    'valor_vendido'    => $vv,
    'premios_pagados'  => $pp,
    'saldo'            => $vv - $pp
  ];
}
mysqli_stmt_close($stmt);

/* ==========================================================
   DETALLE: Sí muestra coordinador (estado=1 o si existe id_coordinador)
   ========================================================== */
$sqlData = "
SELECT
  r.codigo,
  r.estado,
  COALESCE(r.valor,0.50) valor,
  COALESCE(r.premio,0) premio,
  r.fecha_hora,

  CASE
    WHEN COALESCE(r.id_canillita,0) > 0 AND COALESCE(c.canillita,'')<>'' THEN CONCAT('CAN: ', c.canillita)
    WHEN COALESCE(r.id_usuario,0)   > 0 AND COALESCE(u.usuario,'')<>''   THEN CONCAT('PV: ', u.usuario)
    WHEN COALESCE(r.id_coordinador,0) > 0 AND COALESCE(co.nombre,'')<>'' THEN CONCAT('COORD: ', co.nombre)
    ELSE ''
  END responsable
FROM tbr_raspadita r
LEFT JOIN tbr_usuario u        ON u.idusuario=r.id_usuario
LEFT JOIN tbr_canillita c      ON c.id=r.id_canillita
LEFT JOIN tbr_coordinador co   ON co.id=r.id_coordinador
WHERE $where
ORDER BY r.fecha_hora DESC
LIMIT 5000
";
$stmt = mysqli_prepare($db, $sqlData);
if (!$stmt) jexit(['success'=>false,'message'=>'Error prepare detalle: '.mysqli_error($db)], 500);

bind($stmt, $types, $params);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);

$data = [];
while($row=mysqli_fetch_assoc($res)){
  $en = (int)($row['estado'] ?? 0);
  $valor  = numf($row['valor'] ?? 0.50);
  $premio = numf($row['premio'] ?? 0);

  $saldoFila = 0.0;
  if ($en === 8) $saldoFila = $valor - $premio;
  elseif ($en === 3) $saldoFila = $valor;

  $estadoTxt = ($en===1?'ASIGNADA':($en===2?'ENTREGADA':($en===3?'VENDIDA':($en===8?'PAGADA':'ESTADO '.$en))));

  $data[] = [
    'codigo'      => (string)($row['codigo'] ?? ''),
    'estado'      => $estadoTxt,
    'estado_num'  => $en,
    'responsable' => (string)($row['responsable'] ?? ''),
    'valor'       => $valor,
    'premio'      => $premio,
    'saldo_fila'  => $saldoFila,
    'fecha_hora'  => (string)($row['fecha_hora'] ?? '')
  ];
}
mysqli_stmt_close($stmt);

/* ================= RESPUESTA ================= */
jexit([
  'success'=>true,
  'fecha_ini'=>$fecha_ini,
  'fecha_fin'=>$fecha_fin,
  'totales'=>[
    'asignadas'       => $asignadas,
    'valor_asignado'  => $valor_asignado,
    'entregadas'      => $entregadas,
    'valor_entregado' => $valor_entregado,
    'vendidas'        => $vendidas,
    'valor_vendido'   => $valor_vendido,
    'premios_pagados' => $premios_pagados,
    'saldo'           => $saldo
  ],
  'resumen'=>$resumen,
  'data'=>$data
]);