<?php
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;

if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión BD'], JSON_UNESCAPED_UNICODE);
  exit;
}

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

function jexit(array $p, int $c=200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $r = file_get_contents('php://input');
  $j = $r ? json_decode($r, true) : null;
  return is_array($j) ? $j : [];
}
function bind(mysqli_stmt $s, string $t, array &$p): void {
  $b = []; $b[] = $t;
  foreach ($p as $k => &$v) { $b[] = &$v; }
  call_user_func_array([$s, 'bind_param'], $b);
}

$in = read_json();

$fecha_ini = (string)($in['fecha_ini'] ?? date('Y-m-d'));
$fecha_fin = (string)($in['fecha_fin'] ?? date('Y-m-d'));
$id_punto  = (int)($in['id_punto'] ?? 0);
$estado    = trim((string)($in['estado'] ?? ''));

if ($fecha_ini > $fecha_fin) { $tmp=$fecha_ini; $fecha_ini=$fecha_fin; $fecha_fin=$tmp; }

/* ✅ NO mostrar devueltos */
$where = "
  DATE(r.fecha_hora) BETWEEN ? AND ?
  AND COALESCE(r.devuelto_punto,0)=0
  AND COALESCE(r.devuelto_canillita,0)=0
  AND COALESCE(r.devuelto_coordinador,0)=0
";

$params = [$fecha_ini, $fecha_fin];
$types  = "ss";

if ($id_punto > 0) {
  $where .= " AND r.id_usuario = ?";
  $params[] = $id_punto;
  $types .= "i";
}

if ($estado !== '' && ctype_digit($estado)) {
  $est = (int)$estado;
  if (in_array($est, [1,2,3,8], true)) {
    $where .= " AND r.estado = ?";
    $params[] = $est;
    $types .= "i";
  }
}

/* ================= TOTALES (CANTIDAD + VALOR + PREMIO) ================= */
$sqlTot = "
SELECT
  COUNT(*) registros,

  SUM(CASE WHEN r.estado=1 THEN 1 ELSE 0 END) asignadas,
  SUM(CASE WHEN r.estado=2 THEN 1 ELSE 0 END) entregadas,
  SUM(CASE WHEN r.estado=3 THEN 1 ELSE 0 END) vendidas,

  /* ✅ GANADORAS = SOLO ESTADO 8 */
  SUM(CASE WHEN r.estado=8 THEN 1 ELSE 0 END) ganadoras,

  SUM(CASE WHEN r.anulado IN (1,2) THEN 1 ELSE 0 END) bloqueadas,

  /* ✅ SUMA TOTAL VALOR */
  SUM(COALESCE(r.valor,0.50)) total_valor,

  /* ✅ SUMA VALOR POR ESTADO */
  SUM(CASE WHEN r.estado=1 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_asignadas,
  SUM(CASE WHEN r.estado=2 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_entregadas,
  SUM(CASE WHEN r.estado=3 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_vendidas,

  /* ✅ VALOR SOLO PARA ESTADO 8 */
  SUM(CASE WHEN r.estado=8 THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_ganadoras,

  /* ✅ PREMIO SOLO PARA ESTADO 8 */
  SUM(CASE WHEN r.estado=8 THEN COALESCE(r.premio,0) ELSE 0 END) premio_ganadoras,

  /* ✅ VALOR BLOQUEADAS */
  SUM(CASE WHEN r.anulado IN (1,2) THEN COALESCE(r.valor,0.50) ELSE 0 END) valor_bloqueadas

FROM tbr_raspadita r
WHERE $where
";

$stmt = mysqli_prepare($db, $sqlTot);
if(!$stmt) jexit(['success'=>false,'message'=>'Error prepare totales','error'=>mysqli_error($db)], 500);
bind($stmt, $types, $params);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);
$t   = mysqli_fetch_assoc($res) ?: [];
mysqli_stmt_close($stmt);

/* ================= DATA ================= */
$sqlData = "
SELECT
  r.codigo serie,
  r.estado estado_num,

  CASE
    WHEN r.id_canillita>0 AND c.canillita<>'' THEN c.canillita
    WHEN r.id_usuario>0 AND u.usuario<>'' THEN u.usuario
    WHEN r.id_coordinador>0 AND co.nombre<>'' THEN co.nombre
    ELSE ''
  END responsable,

  COALESCE(r.valor,0.50) valor,
  (COALESCE(r.premio,0)>0) ganador,
  COALESCE(r.premio,0) premio,
  CASE WHEN r.anulado IN (1,2) THEN 1 ELSE 0 END bloqueado,
  r.fecha_hora updated_at

FROM tbr_raspadita r
LEFT JOIN tbr_usuario u ON u.idusuario=r.id_usuario
LEFT JOIN tbr_canillita c ON c.id=r.id_canillita
LEFT JOIN tbr_coordinador co ON co.id=r.id_coordinador

WHERE $where
ORDER BY r.fecha_hora DESC
LIMIT 5000
";

$stmt = mysqli_prepare($db, $sqlData);
if(!$stmt) jexit(['success'=>false,'message'=>'Error prepare data','error'=>mysqli_error($db)], 500);
bind($stmt, $types, $params);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);

$data = [];
while($row = mysqli_fetch_assoc($res)) {

  $en = (int)$row['estado_num'];
  $estadoTxt = 'ESTADO '.$en;
  if($en===1) $estadoTxt='ASIGNADA';
  if($en===2) $estadoTxt='ENTREGADA';
  if($en===3) $estadoTxt='VENDIDA';
  if($en===8) $estadoTxt='PAGADA';

  if((int)$row['bloqueado']===1) $estadoTxt='BLOQUEADA';

  $data[] = [
    'serie'       => (string)$row['serie'],
    'estado'      => $estadoTxt,
    'responsable' => (string)$row['responsable'],
    'valor'       => (float)$row['valor'],
    'ganador'     => (int)$row['ganador'],
    'premio'      => (float)$row['premio'],
    'bloqueado'   => (int)$row['bloqueado'],
    'updated_at'  => (string)$row['updated_at'],
  ];
}
mysqli_stmt_close($stmt);

jexit([
  'success' => true,
  'totales' => [
    'registros'         => (int)($t['registros'] ?? 0),
    'asignadas'         => (int)($t['asignadas'] ?? 0),
    'entregadas'        => (int)($t['entregadas'] ?? 0),
    'vendidas'          => (int)($t['vendidas'] ?? 0),

    /* ✅ GANADORAS = estado 8 */
    'ganadoras'         => (int)($t['ganadoras'] ?? 0),

    'bloqueadas'        => (int)($t['bloqueadas'] ?? 0),

    'total_valor'       => (float)($t['total_valor'] ?? 0),

    'valor_asignadas'   => (float)($t['valor_asignadas'] ?? 0),
    'valor_entregadas'  => (float)($t['valor_entregadas'] ?? 0),
    'valor_vendidas'    => (float)($t['valor_vendidas'] ?? 0),

    /* ✅ VALOR + PREMIO SOLO ESTADO 8 */
    'valor_ganadoras'   => (float)($t['valor_ganadoras'] ?? 0),
    'premio_ganadoras'  => (float)($t['premio_ganadoras'] ?? 0),

    'valor_bloqueadas'  => (float)($t['valor_bloqueadas'] ?? 0),
  ],
  'data' => $data
]);