<?php
// views/pages/raspadita/api/raspadita_premio_pagar.php
// POST JSON: { "codigo":"XXXX", "observacion":"...", "id_admin": 1 }
// Reglas:
// - NO anulado=1
// - estado debe ser 3
// - premio > 0
// - NO debe existir pago previo
// - Inserta en tbr_raspadita_pago
// - Actualiza tbr_raspadita.estado = 8

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '[]', true);
  return is_array($j) ? $j : [];
}
function norm(string $s): string {
  $s = trim($s);
  $s = preg_replace('/\s+/', '', $s);
  return $s;
}

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$in = read_json();
$codigo = norm((string)($in['codigo'] ?? ''));
$observacion = trim((string)($in['observacion'] ?? ''));

// id_admin por sesión o JSON
$id_admin_ses = (int)($_SESSION['idusuario'] ?? 0);
$id_admin = $id_admin_ses > 0 ? $id_admin_ses : (int)($in['id_admin'] ?? 0);

if ($id_admin <= 0) jexit(['success'=>false,'message'=>'Sesión no iniciada / id_admin requerido'], 401);
if ($codigo === '') jexit(['success'=>false,'message'=>'codigo requerido'], 422);

$ip  = (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
$now = date('Y-m-d H:i:s');

try{
  $db->begin_transaction();

  // 1) Bloquear registro
  $sqlSel = "
    SELECT id, anulado, estado, COALESCE(premio,0) AS premio
    FROM tbr_raspadita
    WHERE codigo = ?
    LIMIT 1
    FOR UPDATE
  ";
  $st = $db->prepare($sqlSel);
  $st->bind_param("s", $codigo);
  $st->execute();
  $res = $st->get_result();
  $r = $res ? $res->fetch_assoc() : null;
  $st->close();

  if(!$r){
    $db->rollback();
    jexit(['success'=>false,'message'=>'Código no existe'], 404);
  }

  $id_raspadita = (int)$r['id'];
  $anulado = (int)$r['anulado'];
  $estado  = (int)$r['estado'];
  $premio  = (float)$r['premio'];

  if ($anulado === 1){
    $db->rollback();
    jexit(['success'=>false,'message'=>'No se puede pagar: raspadita anulada'], 409);
  }
  if ($estado !== 3){
    $db->rollback();
    jexit(['success'=>false,'message'=>'No se puede pagar: estado debe ser 3 (vendida)'], 409);
  }
  if ($premio <= 0){
    $db->rollback();
    jexit(['success'=>false,'message'=>'No se puede pagar: premio=0'], 409);
  }

  // 2) Validar no pagada
  $sqlPag = "SELECT id FROM tbr_raspadita_pago WHERE codigo = ? LIMIT 1";
  $stp = $db->prepare($sqlPag);
  $stp->bind_param("s", $codigo);
  $stp->execute();
  $rp = $stp->get_result();
  $ya = $rp ? $rp->fetch_assoc() : null;
  $stp->close();

  if($ya){
    $db->rollback();
    jexit(['success'=>false,'message'=>'Esta premiada ya fue pagada'], 409);
  }

  // 3) Insert pago
  $sqlIns = "
    INSERT INTO tbr_raspadita_pago
      (id_raspadita, id_usuario, ip, fecha_hora, observacion, codigo)
    VALUES
      (?, ?, ?, ?, ?, ?)
  ";
  $sti = $db->prepare($sqlIns);
  $sti->bind_param("iissss", $id_raspadita, $id_admin, $ip, $now, $observacion, $codigo);
  $sti->execute();
  $id_pago = (int)$db->insert_id;
  $sti->close();

  // 4) Actualizar estado = 8
  $sqlUp = "UPDATE tbr_raspadita SET estado = 8 WHERE id = ? LIMIT 1";
  $stu = $db->prepare($sqlUp);
  $stu->bind_param("i", $id_raspadita);
  $stu->execute();
  $stu->close();

  $db->commit();

  jexit([
    'success'=>true,
    'message'=>'Pago registrado y estado actualizado a 8',
    'id_pago'=>$id_pago,
    'id_raspadita'=>$id_raspadita,
    'codigo'=>$codigo,
    'estado_nuevo'=>8
  ]);

}catch(Throwable $e){
  try { $db->rollback(); } catch(Throwable $x) {}
  jexit(['success'=>false,'message'=>'Error al registrar pago','detail'=>$e->getMessage()], 500);
}
