<?php
// views/pages/raspadita/api/raspadita_premio_info.php
// POST JSON: { "codigo":"XXXX" }
// Reglas:
// - NO anulado=1
// - estado debe ser 3 (lista para pagar) o 8 (ya pagada)
// - premio > 0
// - devuelve recorrido (coordinador -> punto -> canillita) y si ya está pagada

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '[]', true);
  return is_array($j) ? $j : [];
}
function norm(string $s): string {
  $s = trim($s);
  $s = preg_replace('/\s+/', '', $s);
  return $s;
}

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$in = read_json();
$codigo = norm((string)($in['codigo'] ?? ''));

if ($codigo === '') jexit(['success'=>false,'message'=>'codigo requerido'], 422);

try{
  // 1) Buscar raspadita (NO anulado)
  $sql = "
    SELECT
      r.id,
      r.codigo,
      r.estado,
      r.anulado,
      r.fecha_hora,
      COALESCE(r.premio,0) AS premio,
      COALESCE(r.id_usuario,0) AS id_usuario,
      COALESCE(r.id_canillita,0) AS id_canillita,
      COALESCE(r.id_coordinador,0) AS id_coordinador,

      u.usuario AS punto_nombre,
      c.canillita AS canillita_nombre,
      co.nombre AS coordinador_nombre
    FROM tbr_raspadita r
    LEFT JOIN tbr_usuario u ON u.idusuario = r.id_usuario
    LEFT JOIN tbr_canillita c ON c.id = r.id_canillita
    LEFT JOIN tbr_coordinador co ON co.id = r.id_coordinador
    WHERE r.codigo = ?
    LIMIT 1
  ";
  $st = $db->prepare($sql);
  $st->bind_param("s", $codigo);
  $st->execute();
  $row = ($st->get_result())?->fetch_assoc();
  $st->close();

  if(!$row) jexit(['success'=>false,'message'=>'Código no existe'], 404);

  if ((int)$row['anulado'] === 1) jexit(['success'=>false,'message'=>'No permitido: raspadita está anulada'], 409);

  // Validar premio > 0
  if ((float)$row['premio'] <= 0) {
    jexit(['success'=>false,'message'=>'No es premiada: premio = 0'], 409);
  }

  $estado = (int)$row['estado'];

  // Solo permitimos ver info si está vendida (3) o pagada (8)
  if (!in_array($estado, [3,8], true)) {
    jexit(['success'=>false,'message'=>'No permitido: la raspadita no está vendida/pagada'], 409);
  }

  // 2) Ver si ya fue pagada (por tabla pagos)
  $sqlP = "
    SELECT id, fecha_hora, ip, id_usuario, observacion
    FROM tbr_raspadita_pago
    WHERE codigo = ?
    ORDER BY id DESC
    LIMIT 1
  ";
  $stp = $db->prepare($sqlP);
  $stp->bind_param("s", $codigo);
  $stp->execute();
  $pago = ($stp->get_result())?->fetch_assoc();
  $stp->close();

  // 3) Recorrido (NO mostramos devueltos)
  $cadena = [];
  if ((int)$row['id_coordinador'] > 0 && trim((string)$row['coordinador_nombre']) !== '') {
    $cadena[] = "COORDINADOR: " . (string)$row['coordinador_nombre'];
  }
  if ((int)$row['id_usuario'] > 0 && trim((string)$row['punto_nombre']) !== '') {
    $cadena[] = "PUNTO: " . (string)$row['punto_nombre'];
  }
  if ((int)$row['id_canillita'] > 0 && trim((string)$row['canillita_nombre']) !== '') {
    $cadena[] = "CANILLITA: " . (string)$row['canillita_nombre'];
  }
  if (!count($cadena)) $cadena[] = "—";

  // ✅ Si estado=8, consideramos pagada aunque no exista fila en pagos (por seguridad)
  $ya_pagada = ($estado === 8) ? true : ($pago ? true : false);

  $msg = null;
  if ($estado === 8) $msg = 'Esta raspadita ya se encuentra pagada (estado 8).';

  jexit([
    'success'=>true,
    'message'=>$msg,
    'data'=>[
      'id' => (int)$row['id'],
      'codigo' => (string)$row['codigo'],
      'estado' => $estado,
      'premio' => (float)$row['premio'],
      'fecha_hora' => (string)$row['fecha_hora'],
      'cadena' => $cadena,
      'ya_pagada' => $ya_pagada,
      'pago' => $pago ? [
        'fecha_hora' => (string)($pago['fecha_hora'] ?? ''),
        'ip' => (string)($pago['ip'] ?? ''),
        'observacion' => (string)($pago['observacion'] ?? ''),
      ] : null
    ]
  ]);

}catch(Throwable $e){
  jexit(['success'=>false,'message'=>'Error al consultar','detail'=>$e->getMessage()], 500);
}
