<?php
// views/pages/raspadita/api/raspadita_insert_coord.php
// ASIGNAR A COORDINADOR
// - Si ya existe (anulado=0 y NO devuelta): suma premio y actualiza valor
// - Si no existe: inserta nuevo
// POST JSON: { id_coordinador, codigo, premio, valor }

declare(strict_types=1);

/* ===== FIX opcional sesión corrupta ===== */
if (session_status() !== PHP_SESSION_ACTIVE) {
  $sid = session_id();
  if ($sid !== '' && !preg_match('/^[A-Za-z0-9,-]{1,128}$/', $sid)) {
    @session_id('');
  }
  session_start();
}

date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '[]', true);
  return is_array($j) ? $j : [];
}

/* ===== VALIDAR SESIÓN ===== */
if (!isset($_SESSION['idusuario'])) {
  jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);
}
$id_admin = (int)$_SESSION['idusuario'];

/* ===== BD ===== */
require_once __DIR__ . '/../../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

/* ===== INPUT ===== */
$in = read_json();

$codigo = trim((string)($in['codigo'] ?? ''));
$codigo = preg_replace('/\s+/', '', $codigo);

$id_coordinador = (int)($in['id_coordinador'] ?? 0);
$estado  = isset($in['estado']) ? (int)$in['estado'] : 1;
$anulado = isset($in['anulado']) ? (int)$in['anulado'] : 0;

$premio = round((float)($in['premio'] ?? 0), 2);
$valor  = round((float)($in['valor']  ?? 0.50), 2);

if ($codigo === '') jexit(['success'=>false,'message'=>'Código requerido'], 422);
if ($id_coordinador <= 0) jexit(['success'=>false,'message'=>'Coordinador requerido'], 422);
if (!in_array($anulado, [0,1], true)) jexit(['success'=>false,'message'=>'anulado debe ser 0 o 1'], 422);
if ($premio < 0) jexit(['success'=>false,'message'=>'Premio inválido'], 422);
if ($valor < 0)  jexit(['success'=>false,'message'=>'Valor inválido'], 422);

$fecha_hora = date('Y-m-d H:i:s');
$ip = (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');

try {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
  $db->begin_transaction();

  /* ===== ¿EXISTE YA LA SERIE (NO ANULADA y NO DEVUELTA)? =====
     ✅ FIX: es por codigo, no por id
  */
  $stmtChk = $db->prepare("
    SELECT id, COALESCE(premio,0) AS premio_actual, COALESCE(valor,0) AS valor_actual
    FROM tbr_raspadita
    WHERE codigo = ?
      AND anulado = 0
      AND devuelto_punto IS NULL
      AND devuelto_canillita IS NULL
      AND devuelto_coordinador IS NULL
    ORDER BY id DESC
    LIMIT 1
    FOR UPDATE
  ");
  $stmtChk->bind_param("s", $codigo);
  $stmtChk->execute();
  $res = $stmtChk->get_result();
  $row = $res ? $res->fetch_assoc() : null;
  $stmtChk->close();

  if ($row) {
    // ✅ YA EXISTE: SUMAR PREMIO + actualizar valor y datos
    $idExistente = (int)$row['id'];

    $stmtUp = $db->prepare("
      UPDATE tbr_raspadita
      SET
        premio = COALESCE(premio,0) + ?,
        valor  = ?,
        id_coordinador = ?,
        id_admin = ?,
        fecha_hora = ?,
        ip = ?,
        estado = ?
      WHERE id = ?
        AND anulado = 0
      LIMIT 1
    ");
    // d d i i s s i i
    $stmtUp->bind_param(
      "ddiissii",
      $premio,
      $valor,
      $id_coordinador,
      $id_admin,
      $fecha_hora,
      $ip,
      $estado,
      $idExistente
    );
    $stmtUp->execute();
    $stmtUp->close();

    $db->commit();

    jexit([
      'success' => true,
      'message' => 'Premio aumentado correctamente',
      'mode' => 'update',
      'id' => $idExistente,
      'codigo' => $codigo,
      'premio_sumado' => $premio,
      'valor' => $valor,
      'id_admin' => $id_admin,
      'id_coordinador' => $id_coordinador,
      'fecha_hora' => $fecha_hora
    ]);
  }

  /* ===== NO EXISTE: INSERT NORMAL CON PREMIO + VALOR ===== */
  $stmt = $db->prepare("
    INSERT INTO tbr_raspadita
      (codigo, estado, anulado, fecha_hora, ip, id_admin, id_coordinador, premio, valor)
    VALUES
      (?, ?, ?, ?, ?, ?, ?, ?, ?)
  ");
  // s i i s s i i d d  (9 params)
  $stmt->bind_param(
    "siissiidd",
    $codigo,
    $estado,
    $anulado,
    $fecha_hora,
    $ip,
    $id_admin,
    $id_coordinador,
    $premio,
    $valor
  );

  $stmt->execute();
  $newId = (int)$stmt->insert_id;
  $stmt->close();

  $db->commit();

  jexit([
    'success' => true,
    'message' => 'Asignado a coordinador y premio/valor guardado',
    'mode' => 'insert',
    'id' => $newId,
    'codigo' => $codigo,
    'premio' => $premio,
    'valor' => $valor,
    'id_admin' => $id_admin,
    'id_coordinador' => $id_coordinador,
    'fecha_hora' => $fecha_hora
  ], 201);

} catch (Throwable $e) {
  try { $db->rollback(); } catch(Throwable $x) {}
  jexit([
    'success'=>false,
    'message'=>'Error al guardar',
    'detail'=>$e->getMessage()
  ], 500);
}