<?php
// views/pages/raspadita/api/raspadita_delete.php
// Elimina de tbr_raspadita por codigo + id_canillita y RESTA $0.50 al saldo
// POST JSON: { "codigo":"6766...", "id_canillita": 10, "id_admin": 1 }
//
// ✅ DEVUELVE saldo_nuevo

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== Helpers ===== */
function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $data = json_decode($raw, true);
  return is_array($data) ? $data : [];
}
function norm(string $s): string {
  $s = trim($s);
  $s = preg_replace('/\s+/', '', $s);
  return (string)$s;
}

/* ===== DB ===== */
require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

/* ===== Entrada ===== */
$in = read_json();
$codigo       = norm((string)($in['codigo'] ?? ''));
$id_canillita = (int)($in['id_canillita'] ?? 0);

// ✅ id_admin: sesión o JSON (para curl/postman)
$id_admin_ses = (int)($_SESSION['idusuario'] ?? 0);
$id_admin     = $id_admin_ses > 0 ? $id_admin_ses : (int)($in['id_admin'] ?? 0);

if ($codigo === '') jexit(['success'=>false,'message'=>'Debe enviar "codigo"'], 422);
if ($id_canillita <= 0) jexit(['success'=>false,'message'=>'Debe enviar "id_canillita" (>0)'], 422);
if ($id_admin <= 0) jexit(['success'=>false,'message'=>'Debe enviar "id_admin" (>0) o iniciar sesión'], 401);

$valor = 0.50;

mysqli_begin_transaction($db);

try {
  // ✅ Bloquear canillita para concurrencia (saldo es DECIMAL)
  $stmt = mysqli_prepare($db, "
    SELECT saldo
    FROM tbr_canillita
    WHERE id=? AND activo=1
    LIMIT 1
    FOR UPDATE
  ");
  if (!$stmt) throw new Exception('Error prepare canillita: ' . mysqli_error($db));
  mysqli_stmt_bind_param($stmt, "i", $id_canillita);
  mysqli_stmt_execute($stmt);
  $rc = mysqli_stmt_get_result($stmt);
  $can = mysqli_fetch_assoc($rc) ?: null;
  mysqli_stmt_close($stmt);

  if (!$can) {
    mysqli_rollback($db);
    jexit(['success'=>false,'message'=>'Canillita no existe o está inactivo'], 409);
  }

  // ✅ Buscar entrega exacta (y tomar anulado por seguridad)
  $stmt = mysqli_prepare($db, "
    SELECT id, anulado
    FROM tbr_raspadita
    WHERE codigo=? AND id_canillita=?
    LIMIT 1
    FOR UPDATE
  ");
  if (!$stmt) throw new Exception('Error prepare SELECT raspadita: ' . mysqli_error($db));
  mysqli_stmt_bind_param($stmt, "si", $codigo, $id_canillita);
  mysqli_stmt_execute($stmt);
  $rr = mysqli_stmt_get_result($stmt);
  $row = mysqli_fetch_assoc($rr) ?: null;
  mysqli_stmt_close($stmt);

  if (!$row) {
    mysqli_rollback($db);
    jexit(['success'=>false,'message'=>'No existe esa entrega para ese canillita'], 404);
  }

  $id = (int)$row['id'];
  $anulado = (int)($row['anulado'] ?? 0);

  // ✅ Eliminar entrega
  $stmt = mysqli_prepare($db, "DELETE FROM tbr_raspadita WHERE id=? LIMIT 1");
  if (!$stmt) throw new Exception('Error prepare DELETE: ' . mysqli_error($db));
  mysqli_stmt_bind_param($stmt, "i", $id);
  $ok = mysqli_stmt_execute($stmt);
  $err = mysqli_stmt_error($stmt);
  $aff = mysqli_stmt_affected_rows($stmt);
  mysqli_stmt_close($stmt);

  if (!$ok || $aff <= 0) throw new Exception($err ?: 'No se pudo eliminar');

  // ✅ Restar saldo SOLO si la entrega NO estaba anulada
  // (si tu sistema nunca inserta anulados, igual queda correcto)
  if ($anulado === 0) {
    $stmt = mysqli_prepare($db, "
      UPDATE tbr_canillita
      SET saldo = GREATEST(COALESCE(saldo,0) - ?, 0)
      WHERE id=? AND activo=1
      LIMIT 1
    ");
    if (!$stmt) throw new Exception('Error prepare UPDATE saldo: ' . mysqli_error($db));
    mysqli_stmt_bind_param($stmt, "di", $valor, $id_canillita);
    $ok2 = mysqli_stmt_execute($stmt);
    $err2 = mysqli_stmt_error($stmt);
    $aff2 = mysqli_stmt_affected_rows($stmt);
    mysqli_stmt_close($stmt);

    if (!$ok2 || $aff2 <= 0) throw new Exception($err2 ?: 'No se pudo actualizar saldo');
  }

  // ✅ Consultar saldo nuevo para devolver a la UI
  $stmt = mysqli_prepare($db, "SELECT saldo FROM tbr_canillita WHERE id=? LIMIT 1");
  if (!$stmt) throw new Exception('Error prepare SELECT saldo nuevo: ' . mysqli_error($db));
  mysqli_stmt_bind_param($stmt, "i", $id_canillita);
  mysqli_stmt_execute($stmt);
  $rs2 = mysqli_stmt_get_result($stmt);
  $row2 = mysqli_fetch_assoc($rs2) ?: null;
  mysqli_stmt_close($stmt);

  $saldoNuevo = $row2 ? (float)$row2['saldo'] : 0.0;

  mysqli_commit($db);

  jexit([
    'success'=>true,
    'message'=>'Eliminado y saldo actualizado (-0.50)',
    'id'=>$id,
    'codigo'=>$codigo,
    'id_canillita'=>$id_canillita,
    'valor'=> ($anulado === 0 ? $valor : 0),
    'anulado'=>$anulado,
    'saldo_nuevo'=>$saldoNuevo
  ]);

} catch (Throwable $e) {
  mysqli_rollback($db);
  jexit([
    'success'=>false,
    'message'=>'Error en transacción',
    'detail'=>$e->getMessage(),
    'mysql'=>mysqli_error($db)
  ], 500);
}
