<?php
// ✅ Asigna raspadita a CANILLITA (solo si pertenece al usuario)
// ✅ tbr_raspadita: id_canillita = X, estado = 3
// ✅ tbr_canillita: saldo = saldo + 0.50
// POST JSON: { "codigo":"ABC123", "id_canillita": 10 }

declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = $raw ? json_decode($raw, true) : [];
  return is_array($j) ? $j : [];
}

if (!isset($_SESSION['idusuario'])) jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);
$id_usuario = (int)$_SESSION['idusuario'];

$in = read_json();
$codigo = trim((string)($in['codigo'] ?? ''));
$codigo = preg_replace('/[^\p{L}\p{N}\-_]/u', '', $codigo);
$id_canillita = (int)($in['id_canillita'] ?? 0);

if ($codigo === '' || mb_strlen($codigo) < 4) jexit(['success'=>false,'message'=>'Código inválido'], 400);
if ($id_canillita <= 0) jexit(['success'=>false,'message'=>'Seleccione un canillita'], 400);

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

$VALOR = 0.50;

try {
  $db->begin_transaction();

  // ✅ validar canillita activo
  $canillita_nombre = '';
  $stC = $db->prepare("SELECT canillita FROM tbr_canillita WHERE id=? AND activo=1 LIMIT 1");
  if (!$stC) throw new RuntimeException("Prepare canillita: ".$db->error);
  $stC->bind_param('i', $id_canillita);
  $stC->execute();
  $stC->bind_result($canillita_nombre);
  $okC = $stC->fetch();
  $stC->close();

  if (!$okC) {
    $db->rollback();
    jexit(['success'=>false,'message'=>'Canillita no existe o está inactivo'], 404);
  }

  // ✅ asignar SOLO si pertenece al usuario y está disponible (estado=1)
  $sqlUp = "UPDATE tbr_raspadita
            SET id_canillita = ?,
                estado = 3
            WHERE codigo = ?
              AND id_usuario = ?
              AND anulado = 0
              AND estado = 1
            LIMIT 1";
  $stUp = $db->prepare($sqlUp);
  if (!$stUp) throw new RuntimeException("Prepare asignar: ".$db->error);

  $stUp->bind_param('isi', $id_canillita, $codigo, $id_usuario);
  $stUp->execute();
  $aff = $stUp->affected_rows;
  $stUp->close();

  if ($aff !== 1) {
    $db->rollback();
    jexit(['success'=>false,'message'=>'No se pudo asignar: el código no pertenece a tu usuario o no está estado=1.'], 409);
  }

  // ✅ sumar saldo +0.50
  $sqlSaldo = "UPDATE tbr_canillita
               SET saldo = COALESCE(saldo,0) + ?
               WHERE id = ?
                 AND activo = 1
               LIMIT 1";
  $stS = $db->prepare($sqlSaldo);
  if (!$stS) throw new RuntimeException("Prepare saldo: ".$db->error);

  $stS->bind_param('di', $VALOR, $id_canillita);
  $stS->execute();
  if ($stS->affected_rows !== 1) {
    $stS->close();
    $db->rollback();
    jexit(['success'=>false,'message'=>'No se pudo actualizar el saldo del canillita.'], 500);
  }
  $stS->close();

  // saldo nuevo (opcional)
  $saldoNuevo = null;
  $stGet = $db->prepare("SELECT saldo FROM tbr_canillita WHERE id=? LIMIT 1");
  if ($stGet) {
    $stGet->bind_param('i', $id_canillita);
    $stGet->execute();
    $stGet->bind_result($saldoNuevo);
    $stGet->fetch();
    $stGet->close();
  }

  $db->commit();

  jexit([
    'success'=>true,
    'message'=>"✅ Asignado a ".mb_strtoupper($canillita_nombre)." y saldo +$0.50",
    'data'=>[
      'codigo'=>$codigo,
      'id_canillita'=>$id_canillita,
      'estado'=>3,
      'sumado'=>$VALOR,
      'saldo_nuevo'=>is_null($saldoNuevo) ? null : (float)$saldoNuevo
    ]
  ], 200);

} catch (Throwable $e) {
  try { $db->rollback(); } catch (Throwable $x) {}
  jexit(['success'=>false,'message'=>'Error interno','detail'=>$e->getMessage()], 500);
}
