<?php
// views/pages/raspadita/api/raspadita_asignar.php
// Guarda asignación de raspaditas por punto de venta en tbr_raspadita
// POST JSON: { "id_usuario": 12, "codigo": "ABC123", "premio": 0.00, "valor": 0.50 }

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  jexit(['success'=>false,'message'=>'Método no permitido. Use POST.'], 405);
}

/* ===== Validar sesión ===== */
$id_admin = (int)($_SESSION['idusuario'] ?? 0);
if ($id_admin <= 0) {
  jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);
}

/* ===== Leer JSON ===== */
$raw = file_get_contents('php://input');
$in = json_decode($raw ?: '', true);
if (!is_array($in)) $in = [];

$id_usuario = (int)($in['id_usuario'] ?? 0); // punto de venta
$codigo = trim((string)($in['codigo'] ?? ''));
$codigo = preg_replace('/\s+/', '', $codigo); // sin espacios

// ✅ premio (acepta 0.00)
$premio = (float)($in['premio'] ?? 0);
$premio = round($premio, 2);

// ✅ valor (NUEVO)
$valor = (float)($in['valor'] ?? 0.50);
$valor = round($valor, 2);

if ($id_usuario <= 0) jexit(['success'=>false,'message'=>'id_usuario (punto de venta) es obligatorio'], 422);
if ($codigo === '')   jexit(['success'=>false,'message'=>'codigo (serie) es obligatorio'], 422);
if ($premio < 0)      jexit(['success'=>false,'message'=>'premio no puede ser negativo'], 422);
if ($valor < 0)       jexit(['success'=>false,'message'=>'valor no puede ser negativo'], 422);

/* ===== IP ===== */
$ip = (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
$fecha_hora = date('Y-m-d H:i:s');

/* ===== DB ===== */
require_once __DIR__ . '/../../conex.php';
$db = $conn ?? ($conex ?? null);

if (!$db instanceof mysqli) {
  jexit(['success'=>false,'message'=>'Sin conexión a la base de datos'], 500);
}
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

try {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
  $db->begin_transaction();

  /* ===== Ver si ya existe (activa y NO devuelta) ===== */
  $stmt = $db->prepare("
    SELECT id, COALESCE(premio,0) AS premio_actual, COALESCE(valor,0) AS valor_actual
    FROM tbr_raspadita
    WHERE codigo = ?
      AND anulado = 0
      AND devuelto_punto IS NULL
      AND devuelto_canillita IS NULL
      AND devuelto_coordinador IS NULL
    LIMIT 1
  ");
  $stmt->bind_param("s", $codigo);
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res ? $res->fetch_assoc() : null;
  $stmt->close();

  $estado  = 2;  // asignado a punto
  $anulado = 0;

  if ($row) {
    // ✅ YA EXISTE: SUMAR PREMIO + actualizar valor/punto/admin/fecha/ip/estado
    $id = (int)$row['id'];

    $stmtUp = $db->prepare("
      UPDATE tbr_raspadita
      SET
        premio = COALESCE(premio,0) + ?,
        valor  = ?,              -- ✅ guarda valor
        id_usuario = ?,
        id_admin = ?,
        fecha_hora = ?,
        ip = ?,
        estado = ?
      WHERE id = ?
        AND anulado = 0
        AND devuelto_punto IS NULL
        AND devuelto_canillita IS NULL
        AND devuelto_coordinador IS NULL
      LIMIT 1
    ");
    // d d i i s s i i
    $stmtUp->bind_param("ddiissii", $premio, $valor, $id_usuario, $id_admin, $fecha_hora, $ip, $estado, $id);
    $stmtUp->execute();
    $stmtUp->close();

    $db->commit();

    jexit([
      'success' => true,
      'message' => ($premio > 0 ? 'Premio aumentado' : 'Asignación actualizada'),
      'mode' => 'update',
      'data' => [
        'id' => $id,
        'id_usuario' => $id_usuario,
        'codigo' => $codigo,
        'estado' => $estado,
        'anulado' => $anulado,
        'premio_sumado' => $premio,
        'valor' => $valor,
        'fecha_hora' => $fecha_hora,
        'ip' => $ip,
        'id_admin' => $id_admin
      ]
    ]);
  }

  /* ===== NO EXISTE: INSERT con premio + valor ===== */
  $stmtIns = $db->prepare("
    INSERT INTO tbr_raspadita
      (id_usuario, codigo, estado, anulado, fecha_hora, ip, id_admin, premio, valor)
    VALUES
      (?, ?, ?, ?, ?, ?, ?, ?, ?)
  ");
  // i s i i s s i d d
  $stmtIns->bind_param("isiissidd", $id_usuario, $codigo, $estado, $anulado, $fecha_hora, $ip, $id_admin, $premio, $valor);
  $stmtIns->execute();
  $newId = (int)$stmtIns->insert_id;
  $stmtIns->close();

  $db->commit();

  jexit([
    'success' => true,
    'message' => 'Asignación guardada',
    'mode' => 'insert',
    'data' => [
      'id' => $newId,
      'id_usuario' => $id_usuario,
      'codigo' => $codigo,
      'estado' => $estado,
      'anulado' => $anulado,
      'premio' => $premio,
      'valor' => $valor,
      'fecha_hora' => $fecha_hora,
      'ip' => $ip,
      'id_admin' => $id_admin
    ]
  ]);

} catch (Throwable $e) {
  try { $db->rollback(); } catch(Throwable $x) {}
  jexit(['success'=>false,'message'=>'No se pudo guardar la asignación','detail'=>$e->getMessage()], 500);
}