<?php
declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===================== Helpers ===================== */
function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '', true);
  return is_array($j) ? $j : [];
}
function bind_dynamic(mysqli_stmt $stmt, string $types, array &$params): void {
  $bind = [$types];
  foreach ($params as $k => &$v) { $bind[] = &$v; }
  call_user_func_array([$stmt, 'bind_param'], $bind);
}
function is_date_ymd(string $d): bool {
  return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $d);
}

/* ===================== Seguridad / sesión ===================== */
if (!isset($_SESSION['idusuario'])) {
  jexit(['success'=>false,'message'=>'No autorizado (sesión no iniciada)'], 401);
}
$idusuario = (int)$_SESSION['idusuario'];

/* ===================== DB ===================== */
require_once __DIR__ . '/../../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión a la base de datos'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

/* ===================== Permiso raspadita ===================== */
$raspPerm = 0;
$stmt = mysqli_prepare($db, "SELECT COALESCE(raspadita,0) raspadita FROM tbr_usuario WHERE idusuario=? LIMIT 1");
mysqli_stmt_bind_param($stmt, "i", $idusuario);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);
if ($row = mysqli_fetch_assoc($res)) $raspPerm = (int)$row['raspadita'];
mysqli_stmt_close($stmt);
if ($raspPerm !== 1) jexit(['success'=>false,'message'=>'Acceso denegado'], 403);

/* ===================== Input ===================== */
$in = read_json();

$fecha_ini = trim((string)($in['fecha_ini'] ?? ''));
$fecha_fin = trim((string)($in['fecha_fin'] ?? ''));
$id_punto  = (int)($in['id_punto'] ?? 0);
$estado    = trim((string)($in['estado'] ?? '')); // 1/2/3 o texto

if ($fecha_ini !== '' && !is_date_ymd($fecha_ini)) jexit(['success'=>false,'message'=>'fecha_ini inválida (YYYY-MM-DD)'], 422);
if ($fecha_fin !== '' && !is_date_ymd($fecha_fin)) jexit(['success'=>false,'message'=>'fecha_fin inválida (YYYY-MM-DD)'], 422);

/* ===================== Filtros ===================== */
$where  = "1=1";
$types  = "";
$params = [];

/* Fechas sobre fecha_hora */
if ($fecha_ini !== '' && $fecha_fin !== '') {
  $where .= " AND DATE(r.fecha_hora) BETWEEN ? AND ?";
  $types .= "ss";
  $params[] = $fecha_ini;
  $params[] = $fecha_fin;
} elseif ($fecha_ini !== '') {
  $where .= " AND DATE(r.fecha_hora) >= ?";
  $types .= "s";
  $params[] = $fecha_ini;
} elseif ($fecha_fin !== '') {
  $where .= " AND DATE(r.fecha_hora) <= ?";
  $types .= "s";
  $params[] = $fecha_fin;
}

/* Punto */
if ($id_punto > 0) {
  $where .= " AND r.id_usuario = ?";
  $types .= "i";
  $params[] = $id_punto;
}

/* Estado (1/2/3 o texto) */
if ($estado !== '') {
  if (ctype_digit($estado)) {
    $where .= " AND r.estado = ?";
    $types .= "i";
    $params[] = (int)$estado;
  } else {
    $st = strtoupper($estado);
    $map = ['ASIGNADA'=>1,'ASIGNADO'=>1,'VENDIDA'=>2,'VENDIDO'=>2,'PAGADA'=>3,'PAGADO'=>3];
    if (isset($map[$st])) {
      $where .= " AND r.estado = ?";
      $types .= "i";
      $params[] = (int)$map[$st];
    }
  }
}

/* ===================== SQL (✅ Responsable Punto o Canillita o SIN RESPONSABLE) ===================== */
$sql = "
SELECT
  r.codigo AS serie,
  r.estado AS estado_num,
  CASE r.estado
    WHEN 1 THEN 'ASIGNADA'
    WHEN 2 THEN 'VENDIDA'
    WHEN 3 THEN 'PAGADA'
    ELSE CONCAT('ESTADO ', r.estado)
  END AS estado_txt,

  r.anulado AS anulado_num,
  CASE r.anulado
    WHEN 0 THEN 'OK'
    WHEN 1 THEN 'ELIMINADO'
    WHEN 2 THEN 'BLOQUEADO'
    ELSE CONCAT('ANULADO ', r.anulado)
  END AS anulado_txt,

  r.premio AS premio,
  r.fecha_hora AS updated_at,
  r.id_usuario AS id_punto,
  r.id_canillita AS id_canillita,

  CONCAT(COALESCE(s.sucursal,''),' - ',COALESCE(u.usuario,'')) AS punto_nombre,
  COALESCE(c.canillita,'') AS canillita_nombre,

  /* ✅ responsable SIEMPRE con fallback */
  CASE
    WHEN (r.id_usuario IS NOT NULL AND r.id_usuario <> 0)
      THEN CONCAT(COALESCE(s.sucursal,''),' - ',COALESCE(u.usuario,''))
    WHEN (r.id_canillita IS NOT NULL AND r.id_canillita <> 0 AND COALESCE(c.canillita,'') <> '')
      THEN COALESCE(c.canillita,'')
    ELSE 'SIN RESPONSABLE'
  END AS responsable

FROM tbr_raspadita r
LEFT JOIN tbr_usuario u   ON u.idusuario = r.id_usuario
LEFT JOIN tbr_sucursal s  ON s.idsucursal = u.id_sucursal
LEFT JOIN tbr_canillita c ON c.id = r.id_canillita
WHERE {$where}
ORDER BY r.fecha_hora DESC
LIMIT 3000
";

$stmt = $db->prepare($sql);
if (!$stmt) jexit(['success'=>false,'message'=>'No se pudo preparar SQL','detail'=>$db->error], 500);

if ($types !== '') bind_dynamic($stmt, $types, $params);
if (!$stmt->execute()) jexit(['success'=>false,'message'=>'No se pudo ejecutar','detail'=>$stmt->error], 500);

$res = $stmt->get_result();
if (!$res) jexit(['success'=>false,'message'=>'Tu PHP no soporta get_result (mysqlnd faltante).'], 500);

$rows = [];
while ($r = $res->fetch_assoc()) {

  $anulado_num = (int)($r['anulado_num'] ?? 0);

  $rows[] = [
    'serie'       => (string)$r['serie'],
    'estado'      => (string)$r['estado_txt'],   // texto para UI
    'estado_num'  => (int)$r['estado_num'],
    'anulado'     => (string)$r['anulado_txt'],
    'anulado_num' => $anulado_num,

    // ✅ aquí mantenemos "punto" porque tu UI usa r.punto en la tabla
    'punto'       => (string)$r['responsable'],
    // ✅ y también mandamos "responsable" por si luego quieres cambiar UI
    'responsable' => (string)$r['responsable'],

    'valor'       => '0',
    'ganador'     => 0,
    'premio'      => (string)($r['premio'] ?? '0'),
    'bloqueado'   => ($anulado_num === 2) ? 1 : 0,
    'updated_at'  => (string)($r['updated_at'] ?? ''),
    'id_punto'    => (int)($r['id_punto'] ?? 0),
    'id_canillita'=> (int)($r['id_canillita'] ?? 0),
  ];
}
$stmt->close();

/* ===================== Totales ===================== */
$tot = count($rows);
$asig = 0; $vend = 0; $pag = 0; $bloq = 0; $elim = 0;

foreach ($rows as $x) {
  $en = (int)($x['estado_num'] ?? 0);
  if ($en === 1) $asig++;
  if ($en === 2) $vend++;
  if ($en === 3) $pag++;

  $an = (int)($x['anulado_num'] ?? 0);
  if ($an === 2) $bloq++;
  if ($an === 1) $elim++;
}

jexit([
  'success' => true,
  'data' => $rows,
  'totales' => [
    'registros'  => $tot,
    'asignadas'  => $asig,
    'vendidas'   => $vend,
    'pagadas'    => $pag,
    'bloqueadas' => $bloq,
    'eliminadas' => $elim,
  ],
]);
