<?php
// views/pages/raspadita/api/rasp_report_resumen.php
// POST JSON: { "desde":"YYYY-MM-DD", "hasta":"YYYY-MM-DD" } (opcionales)
// Devuelve totales: ventas (cantidad + $), pagos (cantidad + $), saldo

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '[]', true);
  return is_array($j) ? $j : [];
}
function is_ymd(string $s): bool {
  return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $s);
}

if (!isset($_SESSION['idusuario'])) jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);
$id_usuario = (int)$_SESSION['idusuario'];

$in = read_json();
$desde = trim((string)($in['desde'] ?? ''));
$hasta = trim((string)($in['hasta'] ?? ''));

// si no envían fechas => HOY
if ($desde === '' && $hasta === '') {
  $desde = date('Y-m-d');
  $hasta = date('Y-m-d');
}

// validar
if ($desde !== '' && !is_ymd($desde)) jexit(['success'=>false,'message'=>'desde inválido (YYYY-MM-DD)'], 422);
if ($hasta !== '' && !is_ymd($hasta)) jexit(['success'=>false,'message'=>'hasta inválido (YYYY-MM-DD)'], 422);
if ($desde === '' || $hasta === '') jexit(['success'=>false,'message'=>'Debe enviar desde y hasta, o ninguno (para hoy)'], 422);

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

try{
  // Ventas: usa cabecera tbr_venta_raspadita (si tienes estos campos)
  // Campos asumidos: id_usuario, fecha (datetime), total (decimal), cantidad (int)
  $sqlV = "
    SELECT
      COALESCE(SUM(v.cantidad),0) AS cant_vendidas,
      COALESCE(SUM(v.total),0)    AS total_ventas
    FROM tbr_venta_raspadita v
    WHERE v.id_usuario = ?
      AND DATE(v.fecha) BETWEEN ? AND ?
  ";
  $stV = $db->prepare($sqlV);
  $stV->bind_param('iss', $id_usuario, $desde, $hasta);
  $stV->execute();
  $ventas = ($stV->get_result())?->fetch_assoc() ?: [];
  $stV->close();

  // Pagos: tbr_raspadita_pago + premio desde tbr_raspadita
  // Campos pago: codigo, fecha_hora (datetime), id_usuario (quien pagó)
  $sqlP = "
    SELECT
      COUNT(*) AS cant_pagadas,
      COALESCE(SUM(COALESCE(r.premio,0)),0) AS total_premios
    FROM tbr_raspadita_pago p
    JOIN tbr_raspadita r ON r.codigo = p.codigo
    WHERE p.id_usuario = ?
      AND DATE(p.fecha_hora) BETWEEN ? AND ?
  ";
  $stP = $db->prepare($sqlP);
  $stP->bind_param('iss', $id_usuario, $desde, $hasta);
  $stP->execute();
  $pagos = ($stP->get_result())?->fetch_assoc() ?: [];
  $stP->close();

  $cant_vendidas = (int)($ventas['cant_vendidas'] ?? 0);
  $total_ventas  = (float)($ventas['total_ventas'] ?? 0);

  $cant_pagadas  = (int)($pagos['cant_pagadas'] ?? 0);
  $total_premios = (float)($pagos['total_premios'] ?? 0);

  $saldo = round($total_ventas - $total_premios, 2);

  jexit([
    'success'=>true,
    'data'=>[
      'desde'=>$desde,
      'hasta'=>$hasta,
      'ventas'=>[
        'cantidad'=>$cant_vendidas,
        'total'=>round($total_ventas, 2)
      ],
      'pagos'=>[
        'cantidad'=>$cant_pagadas,
        'total'=>round($total_premios, 2)
      ],
      'saldo'=>$saldo
    ]
  ]);

}catch(Throwable $e){
  jexit(['success'=>false,'message'=>'Error al consultar','detail'=>$e->getMessage()], 500);
}