<?php
// views/pages/raspadita/api/rasp_report_list.php
// POST JSON: { "tipo":"ventas"|"pagos", "desde":"YYYY-MM-DD", "hasta":"YYYY-MM-DD" } (opcionales; si no envías => HOY)
// Devuelve detalle para tablas

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '[]', true);
  return is_array($j) ? $j : [];
}
function is_ymd(string $s): bool {
  return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $s);
}

if (!isset($_SESSION['idusuario'])) jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);
$id_usuario = (int)$_SESSION['idusuario'];

$in = read_json();
$tipo  = trim((string)($in['tipo'] ?? ''));
$desde = trim((string)($in['desde'] ?? ''));
$hasta = trim((string)($in['hasta'] ?? ''));

if ($tipo !== 'ventas' && $tipo !== 'pagos') jexit(['success'=>false,'message'=>'tipo inválido'], 422);

if ($desde === '' && $hasta === '') {
  $desde = date('Y-m-d');
  $hasta = date('Y-m-d');
}

if ($desde !== '' && !is_ymd($desde)) jexit(['success'=>false,'message'=>'desde inválido (YYYY-MM-DD)'], 422);
if ($hasta !== '' && !is_ymd($hasta)) jexit(['success'=>false,'message'=>'hasta inválido (YYYY-MM-DD)'], 422);
if ($desde === '' || $hasta === '') jexit(['success'=>false,'message'=>'Debe enviar desde y hasta, o ninguno (para hoy)'], 422);

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

try{
  if ($tipo === 'ventas') {
    // detalle vendido: tbr_venta_raspadita_detalle + cabecera para fecha
    // asumimos: detalle(id_venta, codigo, valor) y cabecera(id, id_usuario, fecha)
    $sql = "
      SELECT
        DATE_FORMAT(v.fecha, '%Y-%m-%d %H:%i') AS fecha,
        d.codigo AS codigo,
        d.valor  AS valor,
        v.id     AS id_venta
      FROM tbr_venta_raspadita_detalle d
      JOIN tbr_venta_raspadita v ON v.id = d.id_venta
      WHERE v.id_usuario = ?
        AND DATE(v.fecha) BETWEEN ? AND ?
      ORDER BY v.fecha DESC, d.codigo ASC
      LIMIT 5000
    ";
    $st = $db->prepare($sql);
    $st->bind_param('iss', $id_usuario, $desde, $hasta);
    $st->execute();
    $res = $st->get_result();
    $data = [];
    while($r = $res->fetch_assoc()){
      $data[] = [
        'fecha'=>$r['fecha'],
        'codigo'=>$r['codigo'],
        'valor'=>(float)$r['valor'],
        'id_venta'=>(int)$r['id_venta']
      ];
    }
    $st->close();

    jexit(['success'=>true,'data'=>$data, 'desde'=>$desde, 'hasta'=>$hasta]);
  }

  // tipo === pagos
  $sql = "
    SELECT
      DATE_FORMAT(p.fecha_hora, '%Y-%m-%d %H:%i') AS fecha,
      p.codigo AS codigo,
      COALESCE(r.premio,0) AS premio,
      COALESCE(p.observacion,'') AS observacion,
      COALESCE(p.ip,'') AS ip
    FROM tbr_raspadita_pago p
    JOIN tbr_raspadita r ON r.codigo = p.codigo
    WHERE p.id_usuario = ?
      AND DATE(p.fecha_hora) BETWEEN ? AND ?
    ORDER BY p.fecha_hora DESC
    LIMIT 5000
  ";
  $st = $db->prepare($sql);
  $st->bind_param('iss', $id_usuario, $desde, $hasta);
  $st->execute();
  $res = $st->get_result();
  $data = [];
  while($r = $res->fetch_assoc()){
    $data[] = [
      'fecha'=>$r['fecha'],
      'codigo'=>$r['codigo'],
      'premio'=>(float)$r['premio'],
      'observacion'=>$r['observacion'],
      'ip'=>$r['ip']
    ];
  }
  $st->close();

  jexit(['success'=>true,'data'=>$data, 'desde'=>$desde, 'hasta'=>$hasta]);

}catch(Throwable $e){
  jexit(['success'=>false,'message'=>'Error al listar','detail'=>$e->getMessage()], 500);
}