<?php
// views/pages/raspadita/api/punto_saldo_insert.php
// Registra cobro del PUNTO usando tu tabla tbr_coordinador_saldo
// y si tipo_pago = 'CREDITO' => suma total_pagar al saldo del punto en tbr_usuario.saldo
//
// POST JSON:
// {
//   "id_usuario": 12,
//   "num_raspaditas": 10,
//   "subtotal": 5.00,
//   "descuento": 0.50,
//   "total_pagar": 4.50,
//   "tipo_pago": "CREDITO" | "EFECTIVO",
//   "observacion": "opcional",
//   "id_admin": 1 // opcional (si no hay sesión)
// }

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-CSRF-Token');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '', true);
  return is_array($j) ? $j : [];
}

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  jexit(['success'=>false,'message'=>'Método no permitido. Use POST.'], 405);
}

$in = read_json();

$id_admin = (int)($_SESSION['idusuario'] ?? 0);
if ($id_admin <= 0) $id_admin = (int)($in['id_admin'] ?? 0);
if ($id_admin <= 0) jexit(['success'=>false,'message'=>'Sesión no iniciada'], 401);

$id_usuario      = (int)($in['id_usuario'] ?? 0); // punto
$num_raspaditas  = (int)($in['num_raspaditas'] ?? 0);
$subtotal        = round((float)($in['subtotal'] ?? 0), 2);
$descuento       = round((float)($in['descuento'] ?? 0), 2);
$total_pagar     = round((float)($in['total_pagar'] ?? 0), 2);
$tipo_pago       = strtoupper(trim((string)($in['tipo_pago'] ?? 'EFECTIVO')));
$observacion     = trim((string)($in['observacion'] ?? ''));

if ($id_usuario <= 0) jexit(['success'=>false,'message'=>'id_usuario (punto) es obligatorio'], 422);
if ($num_raspaditas <= 0) jexit(['success'=>false,'message'=>'num_raspaditas debe ser > 0'], 422);
if ($subtotal < 0 || $descuento < 0 || $total_pagar < 0) jexit(['success'=>false,'message'=>'Valores no pueden ser negativos'], 422);
if ($tipo_pago !== 'CREDITO' && $tipo_pago !== 'EFECTIVO') $tipo_pago = 'EFECTIVO';

$ip = (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
$fecha_hora = date('Y-m-d H:i:s');

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión a la base de datos'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

try{
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
  $db->begin_transaction();

  // ✅ validar punto existe y traer saldo actual
  $stmtU = $db->prepare("SELECT idusuario, COALESCE(saldo,0) AS saldo FROM tbr_usuario WHERE idusuario=? LIMIT 1");
  $stmtU->bind_param("i", $id_usuario);
  $stmtU->execute();
  $ru = $stmtU->get_result();
  $u = $ru ? $ru->fetch_assoc() : null;
  $stmtU->close();

  if(!$u){
    $db->rollback();
    jexit(['success'=>false,'message'=>'Punto no existe (tbr_usuario)'], 422);
  }

  // ✅ Insert en tu tabla tbr_coordinador_saldo
  // id_coordinador=0, id_canillita=0 porque es cobro PV
  $id_coordinador = 0;
  $id_canillita   = 0;

  $stmtIns = $db->prepare("
    INSERT INTO tbr_coordinador_saldo
      (fecha_hora, num_raspaditas, subtotal, descuento, total_pagar, observacion, id_admin, ip, tipo_pago, id_usuario)
    VALUES
      (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
  ");
  // s i i d d d s i s s i i  =>  "siidddsis sii"  (SIN espacios) => "siidddsis ssii"? NO.
  // ✅ Correcto (12): s i i d d d s i s s i i  => "siidddsis ssii" sin espacios es: "siidddsisssii"
  $stmtIns->bind_param(
    "sidddsissi",
    $fecha_hora,
    
    $num_raspaditas,
    $subtotal,
    $descuento,
    $total_pagar,
    $observacion,
    $id_admin,
    $ip,
    $tipo_pago,
    $id_usuario
    
  );
  $stmtIns->execute();
  $newId = (int)$stmtIns->insert_id;
  $stmtIns->close();

  $saldo_sumado = 0.00;
  $saldo_nuevo  = (float)$u['saldo'];

  // ✅ Si es crédito: suma total_pagar al saldo del punto
  if ($tipo_pago === 'CREDITO') {
    $saldo_sumado = $total_pagar;

    $stmtS = $db->prepare("UPDATE tbr_usuario SET saldo = COALESCE(saldo,0) + ? WHERE idusuario=? LIMIT 1");
    $stmtS->bind_param("di", $saldo_sumado, $id_usuario);
    $stmtS->execute();
    $stmtS->close();

    $stmtSN = $db->prepare("SELECT COALESCE(saldo,0) AS saldo FROM tbr_usuario WHERE idusuario=? LIMIT 1");
    $stmtSN->bind_param("i", $id_usuario);
    $stmtSN->execute();
    $rsn = $stmtSN->get_result();
    $rowS = $rsn ? $rsn->fetch_assoc() : null;
    $stmtSN->close();

    $saldo_nuevo = (float)($rowS['saldo'] ?? $saldo_nuevo);
  }

  $db->commit();

  jexit([
    'success' => true,
    'message' => ($tipo_pago === 'CREDITO' ? 'Cobro registrado y saldo actualizado' : 'Cobro registrado'),
    'id' => $newId,
    'tipo_pago' => $tipo_pago,
    'id_usuario' => $id_usuario,
    'saldo_sumado' => round($saldo_sumado, 2),
    'saldo_nuevo' => round($saldo_nuevo, 2),
  ]);

}catch(Throwable $e){
  try { $db->rollback(); } catch(Throwable $x) {}
  jexit(['success'=>false,'message'=>'No se pudo registrar el cobro','detail'=>$e->getMessage()], 500);
}
