<?php
// raspadita/api/login.php — API LOGIN (JSON) Raspaditas
// Reglas finales:
// - Si existe en tbr_usuario y id_rol=1 (activo=1, condicion=1, raspadita=1) => redirect admin.php
// - Si NO existe en tbr_usuario, busca en tbr_coordinador y si id_rol=7 (activo=1) => redirect admin_coordinador.php
// POST JSON: { "usuario":"...", "clave":"...", "remember": true|false }

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

$method = strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET'));
if ($method === 'OPTIONS') { http_response_code(204); exit; }
if ($method !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== Errores como JSON ===== */
ini_set('display_errors', '0');
error_reporting(E_ALL);

function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

set_error_handler(function($s,$m,$f,$l){
  jexit(['success'=>false,'message'=>'PHP error','detail'=>"$m in $f:$l"], 500);
});
set_exception_handler(function($e){
  jexit(['success'=>false,'message'=>'Exception','detail'=>$e->getMessage()], 500);
});

/* ===== Helpers ===== */
function read_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $j = json_decode($raw, true);
  return is_array($j) ? $j : [];
}
function client_ip(): string {
  return (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
}

/* ===== Input ===== */
$in = read_json();
$usuario  = trim((string)($in['usuario'] ?? ''));
$clave    = trim((string)($in['clave'] ?? ''));
$remember = (bool)($in['remember'] ?? false);

if ($usuario === '' || $clave === '') {
  jexit(['success'=>false,'message'=>'usuario y clave son obligatorios'], 422);
}

/* ===== Cookie params ANTES de session_start ===== */
$secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');

// tiempos
$ttl_short = 30 * 60;          // 30 minutos
$ttl_long  = 8 * 60 * 60;      // 8 horas
$ttl = $remember ? $ttl_long : $ttl_short;

session_set_cookie_params([
  'lifetime' => $ttl,
  'path' => '/',
  'domain' => '',
  'secure' => $secure,
  'httponly' => true,
  'samesite' => 'Lax',
]);

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

/* ===== Rate-limit simple (por sesión) ===== */
$ip  = client_ip();
$key = hash('sha256', strtolower($usuario) . '|' . $ip);
$now = time();

$_SESSION['rl'] = $_SESSION['rl'] ?? [];
$rl = $_SESSION['rl'][$key] ?? ['count'=>0,'ts'=>$now];

$window = 120;     // 2 minutos
$maxAttempts = 8;  // 8 intentos

if (($now - (int)$rl['ts']) > $window) {
  $rl = ['count'=>0,'ts'=>$now];
}
if ((int)$rl['count'] >= $maxAttempts) {
  jexit(['success'=>false,'message'=>'Demasiados intentos. Intenta nuevamente en unos segundos.'], 429);
}

/* ===== Conexión ===== */
require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;

if (!$db instanceof mysqli) {
  jexit(['success'=>false,'message'=>'Sin conexión a la base de datos'], 500);
}

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

/* ============================================================
   1) INTENTA LOGIN COMO ADMIN (tbr_usuario, id_rol=1, raspadita=1)
   ============================================================ */
$adminRow = null;

$sqlU = "SELECT idusuario, usuario, clave, activo, id_rol, condicion, COALESCE(raspadita,0) raspadita
         FROM tbr_usuario
         WHERE usuario = ?
           AND activo = 1
           AND condicion = 1
           AND id_rol = 1
           AND COALESCE(raspadita,0) = 1
         LIMIT 1";

$stmt = mysqli_prepare($db, $sqlU);
if (!$stmt) jexit(['success'=>false,'message'=>'Error preparando SQL (tbr_usuario)'], 500);

mysqli_stmt_bind_param($stmt, "s", $usuario);
mysqli_stmt_execute($stmt);
mysqli_stmt_store_result($stmt);

$foundAdmin = (mysqli_stmt_num_rows($stmt) >= 1);

if ($foundAdmin) {
  mysqli_stmt_bind_result($stmt, $idusuario, $usuario_db, $clave_db, $activo, $id_rol, $condicion, $raspadita_db);
  mysqli_stmt_fetch($stmt);
  $adminRow = [
    'idusuario' => (int)$idusuario,
    'usuario'   => (string)$usuario_db,
    'clave'     => (string)$clave_db,
    'activo'    => (int)$activo,
    'id_rol'    => (int)$id_rol,
    'condicion' => (int)$condicion,
    'raspadita' => (int)$raspadita_db,
  ];
}
mysqli_stmt_close($stmt);

/* ============================================================
   2) SI NO ESTÁ EN tbr_usuario, BUSCA EN tbr_coordinador (id_rol=7)
   ============================================================ */
$coordRow = null;

if (!$foundAdmin) {
  $sqlC = "SELECT id, indentificacion, nombre, activo, saldo, usuario, clave, id_rol
           FROM tbr_coordinador
           WHERE usuario = ?
             AND activo = 1
             AND id_rol = 7
           LIMIT 1";

  $stmt2 = mysqli_prepare($db, $sqlC);
  if (!$stmt2) jexit(['success'=>false,'message'=>'Error preparando SQL (tbr_coordinador)'], 500);

  mysqli_stmt_bind_param($stmt2, "s", $usuario);
  mysqli_stmt_execute($stmt2);
  mysqli_stmt_store_result($stmt2);

  if (mysqli_stmt_num_rows($stmt2) >= 1) {
    mysqli_stmt_bind_result($stmt2, $idc, $ident, $nombre, $activoC, $saldoC, $usuarioC, $claveC, $rolC);
    mysqli_stmt_fetch($stmt2);

    $coordRow = [
      'id'              => (int)$idc,
      'indentificacion' => (string)$ident,
      'nombre'          => (string)$nombre,
      'activo'          => (int)$activoC,
      'saldo'           => (float)$saldoC,
      'usuario'         => (string)$usuarioC,
      'clave'           => (string)$claveC,
      'id_rol'          => (int)$rolC,
    ];
  }
  mysqli_stmt_close($stmt2);
}

/* ===== Si no existe en ninguno ===== */
if (!$adminRow && !$coordRow) {
  $rl['count']++;
  $_SESSION['rl'][$key] = $rl;
  jexit(['success'=>false,'message'=>'Acceso denegado. Usuario no existe o sin permiso.'], 401);
}

/* ============================================================
   VALIDAR CLAVE (hash o texto plano)
   ============================================================ */
$stored = $adminRow ? (string)$adminRow['clave'] : (string)$coordRow['clave'];
$ok = false;

// 1) password_hash()
if ($stored !== '' && password_verify($clave, $stored)) $ok = true;

// 2) fallback texto plano
if (!$ok && $stored !== '' && hash_equals($stored, $clave)) $ok = true;

// 3) fallback texto plano con trim en stored (por si en BD hay espacios)
if (!$ok && $stored !== '' && hash_equals(trim($stored), $clave)) $ok = true;

if (!$ok) {
  $rl['count']++;
  $_SESSION['rl'][$key] = $rl;
  jexit(['success'=>false,'message'=>'Contraseña incorrecta'], 401);
}

/* ===== OK: reset rate-limit ===== */
$_SESSION['rl'][$key] = ['count'=>0,'ts'=>$now];

/* ===== Seguridad sesión ===== */
session_regenerate_id(true);

// Anti-hijacking (UA/IP)
$ua = (string)($_SERVER['HTTP_USER_AGENT'] ?? 'unknown');
$_SESSION['BIND_IP'] = $ip;
$_SESSION['BIND_UA'] = substr($ua, 0, 200);

$_SESSION['ingreso']       = 'YES';
$_SESSION['LAST_ACTIVITY'] = $now;
$_SESSION['EXPIRES_AT']    = $now + $ttl;

/* ============================================================
   RESPUESTA SEGÚN ROL
   ============================================================ */

// ADMIN (id_rol=1)
if ($adminRow && (int)$adminRow['id_rol'] === 1) {
  $_SESSION['idusuario']  = (int)$adminRow['idusuario'];
  $_SESSION['usuario']    = (string)$adminRow['usuario'];
  $_SESSION['id_rol']     = (int)$adminRow['id_rol'];
  $_SESSION['raspadita']  = (int)$adminRow['raspadita'];

  jexit([
    'success'    => true,
    'message'    => 'Login correcto',
    'remember'   => $remember,
    'expires_in' => $ttl,
    'redirect'   => '/rifa_prueba/views/pages/raspadita/admin.php',
    'user' => [
      'id'      => (int)$adminRow['idusuario'],
      'usuario' => (string)$adminRow['usuario'],
      'id_rol'  => (int)$adminRow['id_rol'],
    ]
  ]);
}

// COORDINADOR (id_rol=7)
if ($coordRow && (int)$coordRow['id_rol'] === 7) {
  $_SESSION['id_coordinador'] = (int)$coordRow['id'];
  $_SESSION['usuario']        = (string)$coordRow['usuario'];
  $_SESSION['id_rol']         = (int)$coordRow['id_rol'];
  $_SESSION['nombre']         = (string)$coordRow['nombre'];

  jexit([
    'success'    => true,
    'message'    => 'Login correcto',
    'remember'   => $remember,
    'expires_in' => $ttl,
    'redirect'   => '/rifa_prueba/views/pages/raspadita/admin_coordinador.php',
    'user' => [
      'id_coordinador' => (int)$coordRow['id'],
      'usuario'        => (string)$coordRow['usuario'],
      'id_rol'         => (int)$coordRow['id_rol'],
      'nombre'         => (string)$coordRow['nombre'],
      'saldo'          => (float)$coordRow['saldo'],
    ]
  ]);
}

// Si llega aquí, existe pero no tiene rol permitido
jexit(['success'=>false,'message'=>'Rol no autorizado'], 403);