<?php
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión BD'], JSON_UNESCAPED_UNICODE);
  exit;
}
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

function jexit(array $p, int $c=200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  if(!$raw) return [];
  $j = json_decode($raw, true);
  return is_array($j) ? $j : [];
}
function ip_client(): string {
  $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? ($_SERVER['REMOTE_ADDR'] ?? '');
  if ($ip && strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
  return substr((string)$ip, 0, 45);
}
function money2($v): float {
  $n = (float)$v;
  if (!is_finite($n)) $n = 0.0;
  return round($n, 2);
}

$in = read_json();

$id_coordinador = (int)($in['id_coordinador'] ?? 0);
$num            = (int)($in['num_raspaditas'] ?? 0);
$subtotal       = money2($in['subtotal'] ?? 0);
$descuento      = money2($in['descuento'] ?? 0);
$total_pagar    = money2($in['total_pagar'] ?? 0);
$observacion    = mb_substr(trim((string)($in['observacion'] ?? '')), 0, 250);

// ✅ tipo_pago ahora ES COLUMNA
$tipo_pago = strtoupper(trim((string)($in['tipo_pago'] ?? $in['metodo'] ?? '')));
if (!in_array($tipo_pago, ['EFECTIVO','CREDITO'], true)) $tipo_pago = '';

// id_admin por sesión primero
$id_admin = (int)($_SESSION['idusuario'] ?? 0);
if ($id_admin <= 0) $id_admin = (int)($in['id_admin'] ?? 0);

if ($id_admin <= 0) jexit(['success'=>false,'message'=>'Debe iniciar sesión o enviar id_admin'], 401);
if ($id_coordinador <= 0) jexit(['success'=>false,'message'=>'id_coordinador es obligatorio'], 422);
if ($num <= 0) jexit(['success'=>false,'message'=>'num_raspaditas debe ser > 0'], 422);
if ($tipo_pago === '') jexit(['success'=>false,'message'=>'Tipo pago requerido (EFECTIVO o CREDITO)'], 422);

if ($subtotal < 0 || $descuento < 0 || $total_pagar < 0) {
  jexit(['success'=>false,'message'=>'Valores no pueden ser negativos'], 422);
}

// cuadra total
$calc = round($subtotal - $descuento, 2);
if (abs($calc - $total_pagar) > 0.02) $total_pagar = $calc;

$ip = ip_client();
$fecha_hora = date('Y-m-d H:i:s');

mysqli_begin_transaction($db);

try {
  // 1) Validar coordinador activo + bloquear fila
  $stmt = mysqli_prepare($db, "SELECT id, nombre, saldo FROM tbr_coordinador WHERE id=? AND activo=1 LIMIT 1 FOR UPDATE");
  if (!$stmt) throw new Exception('Error prepare coordinador: ' . mysqli_error($db));
  mysqli_stmt_bind_param($stmt, "i", $id_coordinador);
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);
  $co  = mysqli_fetch_assoc($res) ?: null;
  mysqli_stmt_close($stmt);

  if (!$co) {
    mysqli_rollback($db);
    jexit(['success'=>false,'message'=>'Coordinador no existe o está inactivo'], 409);
  }

  // 2) Insert historial (incluye tipo_pago)
  $sql = "INSERT INTO tbr_coordinador_saldo
          (fecha_hora, id_coordinador, num_raspaditas, subtotal, descuento, total_pagar, observacion, id_admin, ip, tipo_pago)
          VALUES (?,?,?,?,?,?,?,?,?,?)";

  $stmt = mysqli_prepare($db, $sql);
  if (!$stmt) throw new Exception('Error prepare INSERT: ' . mysqli_error($db));

  mysqli_stmt_bind_param(
    $stmt,
    "siidddsiss",
    $fecha_hora,
    $id_coordinador,
    $num,
    $subtotal,
    $descuento,
    $total_pagar,
    $observacion,
    $id_admin,
    $ip,
    $tipo_pago
  );

  $ok  = mysqli_stmt_execute($stmt);
  $err = mysqli_stmt_error($stmt);
  $newId = (int)mysqli_insert_id($db);
  mysqli_stmt_close($stmt);

  if (!$ok) throw new Exception('No se pudo insertar: ' . $err);

  // 3) ✅ SOLO SI ES CREDITO: actualizar saldo coordinador
  if ($tipo_pago === 'CREDITO') {
    $stmt = mysqli_prepare($db, "UPDATE tbr_coordinador SET saldo = COALESCE(saldo,0) + ? WHERE id=? AND activo=1 LIMIT 1");
    if (!$stmt) throw new Exception('Error prepare UPDATE saldo: ' . mysqli_error($db));
    mysqli_stmt_bind_param($stmt, "di", $total_pagar, $id_coordinador);

    $ok2  = mysqli_stmt_execute($stmt);
    $err2 = mysqli_stmt_error($stmt);
    $aff2 = mysqli_stmt_affected_rows($stmt);
    mysqli_stmt_close($stmt);

    if (!$ok2) throw new Exception('No se pudo actualizar saldo: ' . $err2);
    if ($aff2 <= 0) throw new Exception('No se actualizó saldo (coordinador inactivo o no existe).');
  }

  // 4) saldo actual
  $stmt = mysqli_prepare($db, "SELECT saldo FROM tbr_coordinador WHERE id=? LIMIT 1");
  if (!$stmt) throw new Exception('Error prepare SELECT saldo: ' . mysqli_error($db));
  mysqli_stmt_bind_param($stmt, "i", $id_coordinador);
  mysqli_stmt_execute($stmt);
  $r3 = mysqli_stmt_get_result($stmt);
  $row3 = mysqli_fetch_assoc($r3) ?: ['saldo'=>0];
  mysqli_stmt_close($stmt);

  $saldoActual = (float)($row3['saldo'] ?? 0);

  mysqli_commit($db);

  jexit([
    'success' => true,
    'message' => 'Cobro registrado',
    'id' => $newId,
    'fecha_hora' => $fecha_hora,
    'id_coordinador' => $id_coordinador,
    'num_raspaditas' => $num,
    'subtotal' => $subtotal,
    'descuento' => $descuento,
    'total_pagar' => $total_pagar,
    'tipo_pago' => $tipo_pago,
    'saldo_actual' => $saldoActual
  ], 201);

} catch (Throwable $e) {
  mysqli_rollback($db);
  jexit([
    'success'=>false,
    'message'=>'Error al registrar cobro',
    'detail'=>$e->getMessage(),
    'mysql'=>mysqli_error($db)
  ], 500);
}
