<?php
// views/pages/raspadita/api/coordinador_listado.php
// Lista raspaditas del coordinador
// ✅ Default: estados 1,2,3
// ✅ Opcional: incluir_pagadas=true => agrega estado 8
// POST JSON: { id_coordinador, desde?:'YYYY-MM-DD', hasta?:'YYYY-MM-DD', incluir_pagadas?:true }

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? '')) !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }
function read_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $j = json_decode($raw, true);
  return is_array($j) ? $j : [];
}
function clean_date(string $s): string { return preg_match('/^\d{4}-\d{2}-\d{2}$/', $s) ? $s : ''; }

if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES' || (int)($_SESSION['id_rol'] ?? 0) !== 7) {
  jexit(['success'=>false,'message'=>'No autorizado'], 401);
}

$in = read_json();

$id_coordinador = (int)($_SESSION['id_coordinador'] ?? 0);
if ($id_coordinador <= 0 && isset($in['id_coordinador'])) $id_coordinador = (int)$in['id_coordinador'];
if ($id_coordinador <= 0) jexit(['success'=>false,'message'=>'id_coordinador inválido'], 422);

$desde = clean_date(trim((string)($in['desde'] ?? '')));
$hasta = clean_date(trim((string)($in['hasta'] ?? '')));

$incluir_pagadas = (bool)($in['incluir_pagadas'] ?? false);

// ✅ estados dinámicos
$estados = $incluir_pagadas ? [1,2,3,8] : [1,2,3];
$estadoPlaceholders = implode(',', array_fill(0, count($estados), '?'));

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

// max ids
$maxId = 0;
$sqlMax = "SELECT COALESCE(MAX(id),0) FROM tbr_raspadita WHERE id_coordinador=? AND estado IN ($estadoPlaceholders)";
$stmt = mysqli_prepare($db, $sqlMax);
if ($stmt) {
  $types = "i" . str_repeat("i", count($estados));
  $params = array_merge([$id_coordinador], $estados);
  mysqli_stmt_bind_param($stmt, $types, ...$params);
  mysqli_stmt_execute($stmt);
  mysqli_stmt_bind_result($stmt, $mx);
  mysqli_stmt_fetch($stmt);
  mysqli_stmt_close($stmt);
  $maxId = (int)($mx ?? 0);
}

$maxVendidaId = 0;
$stmt = mysqli_prepare($db, "SELECT COALESCE(MAX(id),0) FROM tbr_raspadita WHERE id_coordinador=? AND estado=3");
if ($stmt) {
  mysqli_stmt_bind_param($stmt, "i", $id_coordinador);
  mysqli_stmt_execute($stmt);
  mysqli_stmt_bind_result($stmt, $mv);
  mysqli_stmt_fetch($stmt);
  mysqli_stmt_close($stmt);
  $maxVendidaId = (int)($mv ?? 0);
}

// ✅ SQL dinámico (incluye valor y premio)
$sql = "
SELECT
  r.id,
  r.codigo,
  r.estado,
  r.fecha_hora,
  COALESCE(r.valor,0)  AS valor,
  COALESCE(r.premio,0) AS premio,
  r.id_usuario,
  r.id_canillita,
  COALESCE(c.canillita, '') AS canillita,
  COALESCE(u.usuario, '')   AS usuario_pv
FROM tbr_raspadita r
LEFT JOIN tbr_canillita c ON c.id = r.id_canillita
LEFT JOIN tbr_usuario  u  ON u.idusuario = r.id_usuario
WHERE r.id_coordinador = ?
  AND r.estado IN ($estadoPlaceholders)
";

$params = [$id_coordinador];
$types  = "i";

// estados
foreach ($estados as $e) { $params[] = (int)$e; $types .= "i"; }

if ($desde !== '' && $hasta !== '') {
  $sql .= " AND DATE(r.fecha_hora) BETWEEN ? AND ? ";
  $params[] = $desde;
  $params[] = $hasta;
  $types .= "ss";
}

$sql .= " ORDER BY r.id DESC LIMIT 2000 ";

$stmt = mysqli_prepare($db, $sql);
if (!$stmt) jexit(['success'=>false,'message'=>'Error preparando SQL listado: '.mysqli_error($db)], 500);

mysqli_stmt_bind_param($stmt, $types, ...$params);
mysqli_stmt_execute($stmt);
mysqli_stmt_store_result($stmt);

mysqli_stmt_bind_result(
  $stmt,
  $id,
  $codigo,
  $estado,
  $fecha_hora,
  $valor,
  $premio,
  $id_usuario,
  $id_canillita,
  $canillita,
  $usuario_pv
);

$items = [];
while (mysqli_stmt_fetch($stmt)) {

  $estadoNum = (int)$estado;

  $estadoHtml = match ($estadoNum) {
    1 => '<span class="badge bg-primary">ASIGNADA</span>',
    2 => '<span class="badge bg-warning text-dark">ENTREGADA</span>',
    3 => '<span class="badge bg-success">VENDIDA</span>',
    8 => '<span class="badge" style="background:#a855f7;">PAGADA</span>',
    default => '<span class="badge bg-secondary">DESCONOCIDO</span>',
  };

  $asignado = '—';
  if ((int)$id_canillita > 0 && $canillita !== '') $asignado = 'CANILLITA: ' . $canillita;
  else if ((int)$id_usuario > 0 && $usuario_pv !== '') $asignado = 'PUNTO: ' . $usuario_pv;

  $items[] = [
    'id' => (int)$id,
    'fecha' => (string)$fecha_hora,
    'codigo' => (string)$codigo,
    'valor' => (float)$valor,
    'premio' => (float)$premio,
    'estado' => $estadoHtml,
    'estado_num' => $estadoNum,
    'asignado_a' => $asignado,
    'id_usuario' => (int)$id_usuario,
    'id_canillita' => (int)$id_canillita,
  ];
}
mysqli_stmt_close($stmt);

jexit([
  'success' => true,
  'items' => $items,
  'desde' => $desde,
  'hasta' => $hasta,
  'incluir_pagadas' => $incluir_pagadas,
  'estados' => $estados,
  'max_id' => $maxId,
  'max_vendida_id' => $maxVendidaId
]);