<?php
declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

$method = strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET'));
if ($method === 'OPTIONS') { http_response_code(204); exit; }
if ($method !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE);
  exit;
}

if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') jexit(['success'=>false,'message'=>'Sin sesión'], 401);
if ((int)($_SESSION['id_rol'] ?? 0) !== 7) jexit(['success'=>false,'message'=>'Permiso denegado'], 403);

$idCoord = (int)($_SESSION['id_coordinador'] ?? 0);
if ($idCoord <= 0) jexit(['success'=>false,'message'=>'ID coordinador inválido'], 422);

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

$raw = file_get_contents('php://input');
$in = json_decode($raw ?: '[]', true);
if (!is_array($in)) $in = [];

$codigos = $in['codigos'] ?? [];
if (!is_array($codigos)) $codigos = [];

// Normaliza
$codigos = array_values(array_unique(array_filter(array_map(static function($x){
  $s = trim((string)$x);
  $s = preg_replace('/\s+/', '', $s);
  return $s;
}, $codigos), static fn($x)=>$x !== '')));

if (count($codigos) < 1) {
  jexit(['success'=>true,'items'=>[],'subtotal'=>0,'validos'=>0,'invalidos'=>0,'codigos_validos'=>[]], 200);
}
if (count($codigos) > 500) jexit(['success'=>false,'message'=>'Máximo 500 códigos'], 422);

$ph = implode(',', array_fill(0, count($codigos), '?'));
$types = str_repeat('s', count($codigos));

$sql = "
SELECT codigo, COALESCE(valor,0) valor,
       COALESCE(estado,0) estado,
       COALESCE(anulado,0) anulado,
       COALESCE(id_usuario,0) id_usuario,
       COALESCE(id_canillita,0) id_canillita,
       COALESCE(id_coordinador,0) id_coordinador
FROM tbr_raspadita
WHERE codigo IN ($ph)
";

$stmt = mysqli_prepare($db, $sql);
if (!$stmt) jexit(['success'=>false,'message'=>'Error prepare preview'], 500);

mysqli_stmt_bind_param($stmt, $types, ...$codigos);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);

$rows = [];
while($r = mysqli_fetch_assoc($res)){
  $rows[(string)$r['codigo']] = $r;
}
mysqli_stmt_close($stmt);

$items = [];
$subtotal = 0.0;
$validos = 0;
$invalidos = 0;
$codigos_validos = [];

foreach ($codigos as $c) {

  if (!isset($rows[$c])) {
    $items[] = ['codigo'=>$c,'ok'=>false,'motivo'=>'NO EXISTE'];
    $invalidos++;
    continue;
  }

  $r = $rows[$c];

  // ✅ Validación EXACTA para asignar:
  // - debe ser del coordinador en sesión
  // - estado=1
  // - no anulado
  // - sin destino
  $esMio   = ((int)$r['id_coordinador'] === $idCoord);
  $estado  = (int)$r['estado'];
  $anulado = (int)$r['anulado'];
  $idU     = (int)$r['id_usuario'];
  $idCan   = (int)$r['id_canillita'];
  $valor   = (float)$r['valor'];

  if (!$esMio) {
    $items[] = ['codigo'=>$c,'ok'=>false,'motivo'=>'NO ES TUYO (id_coordinador)'];
    $invalidos++;
    continue;
  }
  if ($anulado === 1) {
    $items[] = ['codigo'=>$c,'ok'=>false,'motivo'=>'ANULADO=1'];
    $invalidos++;
    continue;
  }
  if ($estado !== 1) {
    $items[] = ['codigo'=>$c,'ok'=>false,'motivo'=>"ESTADO=$estado (debe ser 1)"];
    $invalidos++;
    continue;
  }
  if ($idU > 0 || $idCan > 0) {
    $items[] = ['codigo'=>$c,'ok'=>false,'motivo'=>'YA TIENE DESTINO'];
    $invalidos++;
    continue;
  }

  // ✅ OK
  $items[] = ['codigo'=>$c,'ok'=>true,'valor'=>$valor,'motivo'=>'OK'];
  $validos++;
  $subtotal += $valor;
  $codigos_validos[] = $c;
}

jexit([
  'success'=>true,
  'items'=>$items,
  'subtotal'=>$subtotal,
  'validos'=>$validos,
  'invalidos'=>$invalidos,
  'codigos_validos'=>$codigos_validos
], 200);