<?php
// views/pages/raspadita/api/coordinador_cobro.php
// COBRO SIN CAMBIAR ESTADO EN tbr_raspadita
// - Valida que todos sean del coordinador
// - Valida que todos sean estado=3 (VENDIDA)
// - Valida mismo canillita
// - Inserta en tbr_coordinador_saldo
// - Si tipo_pago=credito => incrementa saldo del canillita
// NO actualiza tbr_raspadita.estado

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

function jexit(array $p, int $c=200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }
function read_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $j = json_decode($raw, true);
  return is_array($j) ? $j : [];
}
function ip(): string { return (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0'); }
function now(): string { return date('Y-m-d H:i:s'); }

if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES' || (int)($_SESSION['id_rol'] ?? 0) !== 7) {
  jexit(['success'=>false,'message'=>'No autorizado'], 401);
}

$id_coordinador = (int)($_SESSION['id_coordinador'] ?? 0);
$id_admin = (int)($_SESSION['idusuario'] ?? 0); // si no existe, queda 0
if ($id_coordinador <= 0) jexit(['success'=>false,'message'=>'Sesión sin coordinador'], 401);

$in = read_json();
$ids = $in['ids'] ?? [];
$tipo_pago = strtolower(trim((string)($in['tipo_pago'] ?? '')));
$observacion = trim((string)($in['observacion'] ?? ''));
$descuento = (float)($in['descuento'] ?? 0);

// Precio unitario (CAMBIA AQUÍ si es distinto)
$precio_unitario = (float)($in['precio_unitario'] ?? 0.50);

if (!in_array($tipo_pago, ['efectivo','credito'], true)) jexit(['success'=>false,'message'=>'tipo_pago inválido (efectivo|credito)'], 422);
if (!is_array($ids) || count($ids) < 1) jexit(['success'=>false,'message'=>'ids requerido'], 422);
if (count($ids) > 800) jexit(['success'=>false,'message'=>'Máximo 800 ids'], 422);
if ($precio_unitario <= 0) jexit(['success'=>false,'message'=>'precio_unitario inválido'], 422);
if ($descuento < 0) jexit(['success'=>false,'message'=>'descuento no puede ser negativo'], 422);

$cleanIds = [];
foreach ($ids as $x) { $v = (int)$x; if ($v > 0) $cleanIds[] = $v; }
$cleanIds = array_values(array_unique($cleanIds));
if (count($cleanIds) < 1) jexit(['success'=>false,'message'=>'ids vacíos'], 422);

require_once __DIR__ . '/../../conex.php';
$db = $conn ?? null;
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión BD'], 500);
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

mysqli_begin_transaction($db);

try {
  $ph = implode(',', array_fill(0, count($cleanIds), '?'));
  $types = str_repeat('i', count($cleanIds));

  // 1) leer y validar (bloqueo FOR UPDATE para evitar cobros dobles concurrentes)
  $sqlSel = "
    SELECT id, codigo, estado, COALESCE(id_canillita,0) id_canillita, COALESCE(id_coordinador,0) id_coordinador
    FROM tbr_raspadita
    WHERE id IN ($ph)
    FOR UPDATE
  ";
  $st = mysqli_prepare($db, $sqlSel);
  if (!$st) throw new Exception('Error SQL select');
  mysqli_stmt_bind_param($st, $types, ...$cleanIds);
  mysqli_stmt_execute($st);
  mysqli_stmt_store_result($st);
  mysqli_stmt_bind_result($st, $id, $codigo, $estado, $id_canillita, $id_coord);

  $rows = [];
  while (mysqli_stmt_fetch($st)) {
    $rows[] = [
      'id'=>(int)$id,
      'codigo'=>(string)$codigo,
      'estado'=>(int)$estado,
      'id_canillita'=>(int)$id_canillita,
      'id_coordinador'=>(int)$id_coord
    ];
  }
  mysqli_stmt_close($st);

  if (count($rows) !== count($cleanIds)) {
    mysqli_rollback($db);
    jexit(['success'=>false,'message'=>'Algunos IDs no existen'], 422);
  }

  // Validar: todos del coordinador, estado=3, mismo canillita (>0)
  $can = 0;
  foreach ($rows as $r) {
    if ((int)$r['id_coordinador'] !== $id_coordinador) {
      mysqli_rollback($db);
      jexit(['success'=>false,'message'=>'Hay raspaditas que no pertenecen al coordinador'], 403);
    }
    if ((int)$r['estado'] !== 3) {
      mysqli_rollback($db);
      jexit(['success'=>false,'message'=>'Solo se puede cobrar estado=3 (VENDIDA)'], 409);
    }
    if ((int)$r['id_canillita'] <= 0) {
      mysqli_rollback($db);
      jexit(['success'=>false,'message'=>'Para cobro, deben pertenecer a un CANILLITA'], 409);
    }
    if ($can === 0) $can = (int)$r['id_canillita'];
    if ($can !== (int)$r['id_canillita']) {
      mysqli_rollback($db);
      jexit(['success'=>false,'message'=>'Selecciona solo raspaditas del MISMO canillita para cobrar'], 409);
    }
  }

  $num = count($rows);
  $subtotal = round($num * $precio_unitario, 2);
  $descuento = round(min($descuento, $subtotal), 2);
  $total = round($subtotal - $descuento, 2);

  // 2) insertar cobro
  $sqlIns = "
    INSERT INTO tbr_coordinador_saldo
    (fecha_hora, num_raspaditas, subtotal, descuento, total_pagar, observacion, id_admin, ip, created_at, tipo_pago, id_canillita)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?, ?)
  ";
  $st3 = mysqli_prepare($db, $sqlIns);
  if (!$st3) throw new Exception('Error SQL insert cobro');

  $fh = now();
  $numi = (int)$num;
  $sub = (float)$subtotal;
  $desc = (float)$descuento;
  $tot = (float)$total;
  $obs = (string)$observacion;
  $idad = (int)$id_admin;
  $ipb = (string)ip();
  $tp  = (string)$tipo_pago;
  $idcan = (int)$can;

  mysqli_stmt_bind_param($st3, "sidddssssi", $fh, $numi, $sub, $desc, $tot, $obs, $idad, $ipb, $tp, $idcan);
  mysqli_stmt_execute($st3);
  $newId = (int)mysqli_insert_id($db);
  mysqli_stmt_close($st3);

  // 3) si crédito => subir saldo canillita
  if ($tipo_pago === 'credito') {
    $st4 = mysqli_prepare($db, "UPDATE tbr_canillita SET saldo = COALESCE(saldo,0) + ? WHERE id=? LIMIT 1");
    if (!$st4) throw new Exception('Error SQL update saldo canillita');
    mysqli_stmt_bind_param($st4, "di", $tot, $idcan);
    mysqli_stmt_execute($st4);
    if (mysqli_stmt_affected_rows($st4) < 1) {
      mysqli_stmt_close($st4);
      throw new Exception('No se pudo actualizar saldo del canillita');
    }
    mysqli_stmt_close($st4);
  }

  mysqli_commit($db);

  jexit([
    'success'=>true,
    'message'=>'Cobro registrado (sin cambiar estado)',
    'id_cobro'=>$newId,
    'id_canillita'=>$can,
    'num'=>$num,
    'precio_unitario'=>$precio_unitario,
    'subtotal'=>$subtotal,
    'descuento'=>$descuento,
    'total_pagar'=>$total,
    'tipo_pago'=>$tipo_pago,
    'estado_no_modificado'=>true
  ]);

} catch (Throwable $e) {
  mysqli_rollback($db);
  jexit(['success'=>false,'message'=>'Error', 'detail'=>$e->getMessage()], 500);
}