<?php
// views/pages/raspadita/api/admin_actions.php
declare(strict_types=1);

session_start();
date_default_timezone_set('America/Guayaquil');

/* ===================== HEADERS ===================== */
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

/* ✅ CORS (NO uses '*', ajusta dominios) */
$ORIGIN_OK = [
  'https://latkic.com',
  'https://www.latkic.com',
  'http://localhost',
  'http://localhost:8080'
];
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
if ($origin && in_array($origin, $ORIGIN_OK, true)) {
  header("Access-Control-Allow-Origin: {$origin}");
  header('Vary: Origin');
}
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-CSRF-Token');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Use POST'], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===================== HELPERS ===================== */
function jexit(array $payload, int $code=200): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function read_json(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '', true);
  return is_array($j) ? $j : [];
}
function is_date_ymd(string $d): bool {
  return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $d);
}
function stmt_bind_dynamic(mysqli_stmt $stmt, string $types, array &$params): void {
  $bind = [$types];
  foreach ($params as $k => &$v) { $bind[] = &$v; }
  call_user_func_array([$stmt, 'bind_param'], $bind);
}

/* ===================== SESIÓN / PERMISOS ===================== */
if (!isset($_SESSION['idusuario'])) {
  jexit(['success'=>false,'message'=>'No autorizado (sesión no iniciada)'], 401);
}
$idusuario = (int)$_SESSION['idusuario'];

/* ✅ CSRF recomendado (en admin.php genera $_SESSION['csrf']) */
$csrf = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
if (!isset($_SESSION['csrf']) || !hash_equals((string)$_SESSION['csrf'], (string)$csrf)) {
  jexit(['success'=>false,'message'=>'CSRF inválido'], 403);
}

/* ===================== DB ===================== */
require_once __DIR__ . '/../../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) jexit(['success'=>false,'message'=>'Sin conexión a la base de datos'], 500);

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

/* ✅ Permisos: tbr_usuario(raspadita) */
$raspPerm = 0;
$stmt = mysqli_prepare($db, "SELECT COALESCE(raspadita,0) raspadita FROM tbr_usuario WHERE idusuario=? LIMIT 1");
mysqli_stmt_bind_param($stmt, "i", $idusuario);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);
if ($row = mysqli_fetch_assoc($res)) $raspPerm = (int)$row['raspadita'];
mysqli_stmt_close($stmt);

if ($raspPerm !== 1) jexit(['success'=>false,'message'=>'Acceso denegado'], 403);

/* ===================== INPUT ===================== */
$in = read_json();
$action = trim((string)($in['action'] ?? ''));

if ($action === '') jexit(['success'=>false,'message'=>'action es obligatorio'], 422);

/* =========================================================
   ACTION: reporte (tbr_raspadita)
   estado:
    1 = ASIGNADO
    2 = VENDIDO
    3 = PAGADO

   anulado:
    0 = normal
    1 = eliminado
    2 = bloqueado
========================================================= */
if ($action === 'reporte') {

  $fecha_ini = trim((string)($in['fecha_ini'] ?? ''));
  $fecha_fin = trim((string)($in['fecha_fin'] ?? ''));
  $id_punto  = (int)($in['id_punto'] ?? 0);
  $id_canillita = (int)($in['id_canillita'] ?? 0); // opcional
  $estado_raw = trim((string)($in['estado'] ?? '')); // "1"/"2"/"3"
  $solo_asignados = (int)($in['solo_asignados'] ?? 0); // 1/0

  if ($fecha_ini !== '' && !is_date_ymd($fecha_ini)) jexit(['success'=>false,'message'=>'fecha_ini inválida (YYYY-MM-DD)'], 422);
  if ($fecha_fin !== '' && !is_date_ymd($fecha_fin)) jexit(['success'=>false,'message'=>'fecha_fin inválida (YYYY-MM-DD)'], 422);

  // ✅ WHERE (por defecto: NO mostrar eliminados anulado=1)
  $where  = " (r.anulado IS NULL OR r.anulado <> 1) ";
  $types  = "";
  $params = [];

  // ✅ SOLO ASIGNADOS (tu regla)
  if ($solo_asignados === 1) {
    $where .= " AND r.estado = 1
                AND r.anulado = 0
                AND r.devuelto_punto = 0
                AND r.devuelto_canillita = 0
                AND (
                  (r.id_usuario IS NOT NULL AND r.id_usuario <> 0)
                  OR
                  (r.id_canillita IS NOT NULL AND r.id_canillita <> 0)
                )";
  }

  // ✅ FECHAS
  if ($fecha_ini !== '' && $fecha_fin !== '') {
    $where .= " AND DATE(r.fecha_hora) BETWEEN ? AND ?";
    $types .= "ss";
    $params[] = $fecha_ini;
    $params[] = $fecha_fin;
  } elseif ($fecha_ini !== '') {
    $where .= " AND DATE(r.fecha_hora) >= ?";
    $types .= "s";
    $params[] = $fecha_ini;
  } elseif ($fecha_fin !== '') {
    $where .= " AND DATE(r.fecha_hora) <= ?";
    $types .= "s";
    $params[] = $fecha_fin;
  }

  // ✅ PUNTO
  if ($id_punto > 0) {
    $where .= " AND r.id_usuario = ?";
    $types .= "i";
    $params[] = $id_punto;
  }

  // ✅ CANILLITA
  if ($id_canillita > 0) {
    $where .= " AND r.id_canillita = ?";
    $types .= "i";
    $params[] = $id_canillita;
  }

  // ✅ ESTADO (si no estás en solo_asignados)
  if ($solo_asignados !== 1 && $estado_raw !== '') {
    if (!ctype_digit($estado_raw)) jexit(['success'=>false,'message'=>'estado inválido (1/2/3)'], 422);
    $est = (int)$estado_raw;
    if (!in_array($est, [1,2,3], true)) jexit(['success'=>false,'message'=>'estado inválido (1/2/3)'], 422);

    $where .= " AND r.estado = ?";
    $types .= "i";
    $params[] = $est;
  }

  /* ======================================================
     ✅ SQL CORREGIDO:
     - LEFT JOIN tbr_canillita c
     - Punto sin " - " fantasma: CONCAT_WS + NULLIF
     - Responsable: Punto si hay, si no Canillita, si no SIN RESPONSABLE
  ====================================================== */
  $sql = "
    SELECT
      r.codigo AS serie,
      r.estado AS estado_num,
      r.anulado AS anulado_num,
      COALESCE(r.premio,0) AS premio,
      r.fecha_hora AS updated_at,
      r.id_usuario AS id_punto,
      r.id_canillita,
      r.devuelto_punto,
      r.devuelto_canillita,

      /* Punto real (sin guión cuando faltan datos) */
      TRIM(CONCAT_WS(' - ', NULLIF(s.sucursal,''), NULLIF(u.usuario,''))) AS punto_nombre,

      /* Canillita */
      COALESCE(NULLIF(c.canillita,''), '') AS canillita_nombre,

      /* ✅ Responsable final */
      CASE
        WHEN (r.id_usuario IS NOT NULL AND r.id_usuario <> 0)
          THEN TRIM(CONCAT_WS(' - ', NULLIF(s.sucursal,''), NULLIF(u.usuario,'')))
        WHEN (r.id_canillita IS NOT NULL AND r.id_canillita <> 0)
          THEN COALESCE(NULLIF(c.canillita,''), CONCAT('CANILLITA #', r.id_canillita))
        ELSE 'SIN RESPONSABLE'
      END AS responsable

    FROM tbr_raspadita r
    LEFT JOIN tbr_usuario u   ON u.idusuario  = r.id_usuario
    LEFT JOIN tbr_sucursal s  ON s.idsucursal = u.id_sucursal
    LEFT JOIN tbr_canillita c ON c.id = r.id_canillita
    WHERE {$where}
    ORDER BY r.fecha_hora DESC
    LIMIT 3000
  ";

  $stmt = mysqli_prepare($db, $sql);
  if (!$stmt) jexit(['success'=>false,'message'=>'No se pudo preparar SQL','detail'=>$db->error], 500);
  if ($types !== '') stmt_bind_dynamic($stmt, $types, $params);

  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);
  if (!$res) jexit(['success'=>false,'message'=>'Tu PHP no soporta get_result (mysqlnd faltante).'], 500);

  $rows = [];
  while ($r = mysqli_fetch_assoc($res)) {
    $estadoNum  = (int)($r['estado_num'] ?? 0);
    $anuladoNum = (int)($r['anulado_num'] ?? 0);

    $estadoTxt = match ($estadoNum) {
      1 => 'ASIGNADO',
      2 => 'VENDIDO',
      3 => 'PAGADO',
      default => (string)$estadoNum
    };

    $bloq = ($anuladoNum === 2) ? 1 : 0;

    $rows[] = [
      'serie'      => (string)($r['serie'] ?? ''),
      'estado'     => $estadoTxt,

      /* ✅ ESTA ES LA CLAVE: tu tabla usa r.punto -> aquí va responsable */
      'punto'      => (string)($r['responsable'] ?? 'SIN RESPONSABLE'),
      'responsable'=> (string)($r['responsable'] ?? 'SIN RESPONSABLE'),

      'valor'      => '0',
      'ganador'    => 0,
      'premio'     => (string)($r['premio'] ?? '0'),
      'bloqueado'  => $bloq,
      'updated_at' => (string)($r['updated_at'] ?? ''),
      'id_punto'   => (int)($r['id_punto'] ?? 0),
      'id_canillita'       => (int)($r['id_canillita'] ?? 0),
      'devuelto_punto'     => (int)($r['devuelto_punto'] ?? 0),
      'devuelto_canillita' => (int)($r['devuelto_canillita'] ?? 0),

      'estado_num' => $estadoNum,
      'anulado'    => $anuladoNum
    ];
  }
  mysqli_stmt_close($stmt);

  // ✅ KPIs
  $tot = count($rows);
  $asig = 0; $vend = 0; $pag = 0; $bloq = 0;

  foreach ($rows as $x) {
    $st = (int)($x['estado_num'] ?? 0);
    if ($st === 1) $asig++;
    if ($st === 2) $vend++;
    if ($st === 3) $pag++;
    if ((int)($x['anulado'] ?? 0) === 2) $bloq++;
  }

  jexit([
    'success' => true,
    'data' => $rows,
    'totales' => [
      'registros'  => $tot,
      'asignadas'  => $asig - $bloq,
      'entregadas' => 0,
      'vendidas'   => $vend,
      'ganadoras'  => 0,
      'pagadas'    => $pag,
      'bloqueadas' => $bloq,
    ],
  ]);
}

/* ===================== DEFAULT ===================== */
jexit(['success'=>false,'message'=>'Acción no válida'], 400);
