<?php
// pago.php — Aplicar pagos a tickets ganadores (muestra PENDIENTES y PAGADOS)
// Requiere: buscar_ticket.php y guardar_pago.php
declare(strict_types=1);
session_start();
include("conex.php");
date_default_timezone_set('America/Guayaquil');

/* ===== Validación de sesión ===== */
if (!isset($_SESSION['idusuario'], $_SESSION['usuario'], $_SESSION['sucursal'])) {
    header('Location: index.php'); exit();
}
$idusuario   = (int) $_SESSION['idusuario'];
$usuario     = htmlspecialchars((string) $_SESSION['usuario']);
$sucursal_id = $_SESSION['sucursal'] ?? null;
if (!is_numeric($sucursal_id)) { die("ID de sucursal inválido"); }
$sucursal_id = (int)$sucursal_id;

/* ===== Conexión ===== */
if (!isset($conn) || !$conn) { die("Error: No se pudo establecer la conexión a la base de datos."); }

/* ===== Datos de sucursal ===== */
$sucursal = '';
if ($stmt = mysqli_prepare($conn, "SELECT sucursal FROM tbr_sucursal WHERE idsucursal = ?")) {
  mysqli_stmt_bind_param($stmt, "i", $sucursal_id);
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);
  if ($row = mysqli_fetch_assoc($res)) { $sucursal = $row['sucursal']; }
  else { die("Sucursal no encontrada."); }
  mysqli_stmt_close($stmt);
} else { die("Error en la preparación de la consulta (sucursal)."); }

/* ===== Permisos ===== */
$venta = 0; $pago = 0;
if ($stmtu = mysqli_prepare($conn, "SELECT venta, pago FROM tbr_usuario WHERE idusuario = ?")) {
  mysqli_stmt_bind_param($stmtu, "i", $idusuario);
  mysqli_stmt_execute($stmtu);
  $resu = mysqli_stmt_get_result($stmtu);
  if ($row = mysqli_fetch_assoc($resu)) { $venta = (int)$row['venta']; $pago = (int)$row['pago']; }
  else { die("Usuario no encontrado"); }
  mysqli_stmt_close($stmtu);
} else { die("Error en la preparación de la consulta (usuario)."); }
?>
<!doctype html>
<html lang="es">
<head>
<meta charset="utf-8">
<title>Sistema de Gestión - Pago</title>
<meta name="viewport" content="width=device-width,initial-scale=1">
<link rel="shortcut icon" href="../resources/img/crud.ico">

<!-- CSS/JS -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css" rel="stylesheet">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-select@1.14.0-beta2/dist/css/bootstrap-select.min.css">

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/dayjs/1.11.10/dayjs.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/dayjs/1.11.10/locale/es.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<style>
  body{background:#f4f6f9}
  .sidebar{width:150px;background:#343a40;padding:20px;height:100vh;position:fixed;left:0;top:0;display:flex;flex-direction:column;align-items:center;overflow:auto}
  .sidebar a{color:#fff;text-decoration:none;padding:12px;width:100%;display:flex;flex-direction:column;align-items:center;transition:background .2s,transform .15s}
  .sidebar a:hover{background:rgba(255,255,255,.1);transform:scale(1.04);border-radius:8px}
  .sidebar .active{font-weight:bold;color:#17a2b8}
  .main-content{margin-left:0;transition:margin-left .3s}
  @media(min-width:769px){.main-content{margin-left:150px}}
  th:first-child, td:first-child{display:none} /* ocultar id */
  .form-control, .btn, select.form-control{font-size:1.1rem;min-height:2.6rem}
</style>
</head>
<body>

<!-- Sidebar -->
<nav class="sidebar" id="sidebar">
  <ul class="nav flex-column text-center w-100">
    <li class="nav-item"><a href="dashboard.php"><i class="fa fa-tachometer-alt"></i> Dashboard</a></li>
    <?php if ($venta === 1): ?>
    <li class="nav-item">
      <a href="ventas.php" class="nav-link <?php echo basename($_SERVER['PHP_SELF'])=='ventas.php'?'active':'';?>">
        <i class="fas fa-shopping-cart fa-2x"></i><div class="text-uppercase" style="font-size:12px">Ventas</div>
      </a>
    </li>
    <?php endif; ?>
    <?php if ($pago === 1): ?>
    <li class="nav-item">
      <a href="pago.php" class="nav-link <?php echo basename($_SERVER['PHP_SELF'])=='pago.php'?'active':'';?>">
        <i class="fas fa-money-check-alt fa-2x"></i><div class="text-uppercase" style="font-size:12px">Pago</div>
      </a>
    </li>
    <?php endif; ?>
    <li class="nav-item"><a href="historial.php" class="nav-link"><i class="fa fa-history fa-2x"></i><div class="text-uppercase" style="font-size:12px">Historial</div></a></li>
    <li class="nav-item"><a href="reporte_usu2.php" class="nav-link"><i class="fa fa-cubes fa-2x"></i><div class="text-uppercase" style="font-size:12px">Utilidad dos cifras</div></a></li>
    <li class="nav-item"><a href="reporte_usu3.php" class="nav-link"><i class="fa fa-cubes fa-2x"></i><div class="text-uppercase" style="font-size:12px">Utilidad tres cifras</div></a></li>
    <li class="nav-item"><a href="reporte_usu4.php" class="nav-link"><i class="fa fa-cubes fa-2x"></i><div class="text-uppercase" style="font-size:12px">Utilidad cuatro cifras</div></a></li>
    <li class="nav-item"><a href="javascript:void(0)" class="nav-link" onclick="cerrar();"><i class="fas fa-sign-out-alt fa-2x"></i><div class="text-uppercase" style="font-size:12px">Cerrar Sesión</div></a></li>
  </ul>
</nav>

<!-- Contenido -->
<div class="main-content" id="mainContent">
  <nav class="navbar navbar-expand-lg navbar-light bg-light">
    <div class="container-fluid">
      <button class="navbar-toggler" type="button" onclick="toggleSidebar()"><span class="navbar-toggler-icon"></span></button>
      <a class="navbar-brand" href="#"><i class="fas fa-tachometer-alt"></i> Panel de pagos</a>
      <div class="collapse navbar-collapse">
        <ul class="navbar-nav ms-auto">
          <li class="nav-item"><a class="nav-link active"><i class="fa fa-store"></i> <?php echo " ".$sucursal; ?></a></li>
          <li class="nav-item"><a class="nav-link active"><i class="fa fa-user-secret"></i> <?php echo " ".$usuario; ?></a></li>
          <li class="nav-item"><a class="nav-link"><i class="fa fa-calendar-check"></i> <span id="fechaActual"></span></a></li>
          <li class="nav-item d-none d-md-block"><a class="nav-link"><em class="fas fa-clock text-mac"></em>&nbsp;<span id="horaActual"></span></a></li>
        </ul>
      </div>
    </div>
  </nav>

  <ul class="nav">
    <?php $current_page = basename($_SERVER['PHP_SELF']); ?>
    <?php if ($venta === 1): ?>
      <li class="nav-item"><a class="nav-link <?php echo ($current_page=='ventas.php')?'active':'';?>" href="ventas.php">Nueva Venta</a></li>
    <?php endif; ?>
    <?php if ($pago === 1): ?>
      <li class="nav-item"><a class="nav-link <?php echo ($current_page=='pago.php')?'active':'';?>" href="pago.php">Aplicar Pago</a></li>
    <?php endif; ?>
  </ul>

  <div class="content p-3">
    <div class="content-heading mb-3">
      <h3 class="m-0">Aplicar Pago <i class="fas fa-file-invoice-dollar fa-1x text-primary"></i></h3>
    </div>

    <div class="row">
      <!-- Filtros -->
      <div class="col-12 col-md-5 col-lg-4">
        <div class="card border-primary mb-3">
          <div class="card-body">
            <div class="mb-3">
              <label class="form-label">Fecha</label>
              <div class="input-group">
                <span class="input-group-text bg-transparent"><i class="fas fa-calendar-week fa-lg"></i></span>
                <input name="fecha" id="fecha" type="date" class="form-control" required>
              </div>
            </div>
            <div class="mb-3">
              <label class="form-label">Lotería</label>
              <div class="input-group">
                <span class="input-group-text bg-transparent"><i class="fas fa-play fa-lg"></i></span>
                <select name="loterias" id="loterias" class="form-control" required>
                  <option value="">Seleccione...</option>
                </select>
              </div>
            </div>
            <div class="mb-3">
              <label class="form-label">Key</label>
              <div class="input-group">
                <span class="input-group-text bg-transparent"><i class="fas fa-key fa-lg"></i></span>
                <input class="form-control" name="key" id="key" type="text" placeholder="Key" required>
              </div>
            </div>
            <div class="mb-2">
              <button class="btn btn-info btn-lg w-100" onclick="buscar()" id="buscar"><i class="fa fa-search"></i> Buscar</button>
            </div>
          </div>
        </div>
      </div>

      <!-- Tabla -->
      <div class="col-12 col-md-7 col-lg-8">
        <div class="table-responsive">
          <table id="resultadosTabla" class="table table-striped table-bordered table-hover">
            <thead class="table-light">
              <tr>
                <th>id</th>
                <th>Código</th>
                <th>Loteria</th>
                <th>Número</th>
                <th>Suerte</th>
                <th>A pagar</th>
              </tr>
            </thead>
            <tbody></tbody>
            <tfoot>
              <tr>
                <th></th>
                <td colspan="5" style="text-align:center" id="botonPagoContainer"></td>
              </tr>
            </tfoot>
          </table>
          <div class="small text-muted mt-2">
            <span class="badge bg-warning text-dark">PENDIENTE</span>
            <span class="badge bg-success">PAGADO</span>
          </div>
        </div>
      </div>

    </div>
  </div>
</div>

<script>
function cerrar(){
  Swal.fire({
    title:'¿Estás seguro?',
    text:'¿Deseas cerrar sesión?',
    icon:'warning',
    showCancelButton:true,
    confirmButtonText:'Sí, cerrar sesión',
    cancelButtonText:'Cancelar'
  }).then((r)=>{
    if(r.isConfirmed){
      const loader=document.createElement('div');
      loader.className='loader';
      loader.textContent='Cerrando sesión...';
      document.body.appendChild(loader);

      fetch('../../controllers/usuario.php',{
        method:'POST',
        headers:{'Content-Type':'application/x-www-form-urlencoded'},
        body:new URLSearchParams({boton:'cerrar'})
      })
      .then((res)=>{ if(!res.ok) throw new Error('Error en la solicitud'); return res.text(); })
      .then(()=>location.href='../index.php')
      .catch((e)=>Swal.fire('Error',e.message,'error'))
      .finally(()=>loader.remove());
    }
  });
}

function toggleSidebar(){
  document.getElementById('sidebar').classList.toggle('show');
}

document.addEventListener('DOMContentLoaded', function(){
  dayjs.locale('es');

  const hoy = new Date();
  const haceOcho = new Date();
  haceOcho.setDate(hoy.getDate()-8);
  const f = (d)=> d.toISOString().split('T')[0];

  document.getElementById('fecha').max = f(hoy);
  document.getElementById('fecha').min = f(haceOcho);

  document.getElementById('fechaActual').textContent = dayjs().format('dddd, DD MMMM YYYY');
  document.getElementById('horaActual').textContent  = dayjs().format('h:mm A');
  setInterval(()=>{ document.getElementById('horaActual').textContent=dayjs().format('h:mm A') },60000);

  document.getElementById('fecha').addEventListener('change', function(){
    const fecha=this.value;
    const $sel=$('#loterias');
    $sel.html('<option value="">Cargando...</option>');
    $.ajax({type:'POST',url:'datos.php?Accion=GetLoteriasP&fechap='+encodeURIComponent(fecha)})
      .done((resp)=>{ $sel.html(resp); document.getElementById('key').focus(); })
      .fail(()=>{ $sel.html('<option value="">Error al cargar datos</option>'); });
  });

  document.addEventListener('keydown', function(e){
    if(e.key==='Enter'){ buscar(); }
  });
});

function buscar(){
  const fecha    = $('#fecha').val();
  const loterias = $('#loterias').val();
  const key      = $('#key').val();

  if(!fecha || !loterias || !key){
    Swal.fire('Error','Hay campos sin llenar o con valores incorrectos.','error');
    return;
  }

  $.ajax({
    url:'buscar_ticket.php',
    type:'POST',
    dataType:'json',
    data:{fecha,loterias,key}
  })
  .done(function(data){
    const $tbody = $('#resultadosTabla tbody'); $tbody.empty();
    const $btnCell = $('#resultadosTabla tfoot tr td#botonPagoContainer'); $btnCell.empty();

    if(data && data.error){
      Swal.fire('¡Hoy no has ganado!', data.error, 'info');
      $('#key').val('');
      return;
    }

    if(!Array.isArray(data) || data.length===0){
      Swal.fire('Sin resultados','No se encontraron resultados.','info');
      return;
    }

    let sumaPremios = 0.0;
    let allIds = [];

    data.forEach(function(p){
      const estado_t = Number(p.estado_t || 0);
      const esPendiente = (estado_t === 2);
      const esPagado    = (estado_t === 3);

      if (esPendiente || esPagado) {
        const tr = document.createElement('tr');
        if (esPagado) tr.classList.add('table-success');
        if (esPendiente) tr.classList.add('table-warning');

        const premioTxt = esPagado
          ? `${p.premio || '0.00'} (PAGADO)`
          : (p.premio || '0.00');

        const cols = [
          p.idrifa || '',
          p.codigo || '-',
          $('#loterias option:selected').text() || (p.loteria || '-'),
          p.numero_ganador || 'N/A',
          p.suerte_ga || 'N/A',
          premioTxt
        ];

        cols.forEach((txt)=>{
          const td=document.createElement('td');
          td.innerHTML = txt;
          tr.appendChild(td);
        });

        $tbody.append(tr);

        // ✅ SOLO cobrar pendientes
        if (esPendiente) {
          const monto = parseFloat(p.premio || '0');
          if(!isNaN(monto)) sumaPremios += monto;

          // ✅ OJO: lo ideal es usar p.ids_pendientes si ya actualizaste buscar_ticket.php
          if (p.ids_pendientes) {
            p.ids_pendientes.split(',').forEach(id=>{
              id=(id || '').trim();
              if(id && !allIds.includes(id)) allIds.push(id);
            });
          } else if (p.ids) {
            p.ids.split(',').forEach(id=>{
              id=(id || '').trim();
              if(id && !allIds.includes(id)) allIds.push(id);
            });
          }
        }
      }
    });

    const total = sumaPremios.toFixed(2);

    if(allIds.length > 0){
      const btn = document.createElement('button');
      btn.className = 'btn btn-success';
      btn.textContent = 'Pagar $' + total;

      btn.addEventListener('click', function(){
        Swal.fire({
          title:'Confirmar pago',
          html:`Se procesarán <b>${allIds.length}</b> ticket(s) pendientes por un total de <b>$${total}</b>.`,
          icon:'warning',
          showCancelButton:true,
          confirmButtonText:'Sí, pagar',
          cancelButtonText:'Cancelar'
        }).then(async (res)=>{
          if(!res.isConfirmed) return;

          btn.disabled = true;

          const errores = [];

          Swal.fire({
            title: 'Procesando pagos...',
            html: 'No cierres esta ventana.',
            allowOutsideClick: false,
            allowEscapeKey: false,
            didOpen: () => Swal.showLoading()
          });

          for (const idrifa of allIds) {
            try {
              const resp = await $.ajax({
                url: 'guardar_pago.php',
                type: 'POST',
                dataType: 'json',
                data: { idrifa }
              });

              if (!(resp && resp.success)) {
                // si no fue success, lo marcamos como error (incluye already_paid/not_payable)
                errores.push(idrifa);
              }
            } catch (e) {
              errores.push(idrifa);
            }
          }

          Swal.close();

          // ✅ SIN RESUMEN, solo mensaje simple
          if (errores.length > 0) {
            Swal.fire({
              title: 'Atención',
              text: 'Algunos tickets no se pudieron pagar.',
              icon: 'warning',
              confirmButtonText: 'Aceptar'
            }).then(()=> location.reload());
            return;
          }

          Swal.fire({
            title: 'Pago completado',
            text: 'Operación realizada correctamente.',
            icon: 'success',
            confirmButtonText: 'Aceptar'
          }).then(()=> location.reload());
        });
      });

      $btnCell.append(btn);
    } else {
      $btnCell.innerHTML = '<span class="text-muted">No hay tickets pendientes para pagar.</span>';
    }
  })
  .fail(function(xhr){
    console.error('Error AJAX:', xhr.responseText || xhr.statusText);
    Swal.fire('Error','Ocurrió un problema al buscar.','error');
  });
}
</script>

</body>
</html>
