<?php
// guardar_ticket.php — Inserta ticket (2 cifras) y responde JSON limpio
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Chequeo de sesión ===== */
if (!isset($_SESSION['idusuario'], $_SESSION['sucursal'])) {
  echo json_encode(['success'=>false,'error'=>'Sesión no iniciada.']); exit;
}
$id_usuario   = (int)$_SESSION['idusuario'];
$sucursal_id  = (int)$_SESSION['sucursal'];
$fecha_hoy    = date('Y-m-d');
$hora_hoy_hms = date('H:i:s');
$hora_hoy_Hi  = (int)date('Hi'); // 14:35 -> 1435

/* ===== Conexión ===== */
require __DIR__ . '/conex.php';
$cn = $conn ?? ($conex ?? null);
if (!$cn instanceof mysqli) {
  echo json_encode(['success'=>false,'error'=>'No hay conexión a la BD']); exit;
}
if (function_exists('mysqli_report')) {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
}
mysqli_set_charset($cn, 'utf8mb4');

/* ===== Helpers ===== */
function post_str(string $k): string {
  return isset($_POST[$k]) ? trim((string)$_POST[$k]) : '';
}
function to_2d_number(string $raw): string {
  // deja solo dígitos, toma los últimos 2 y left-pad con 0
  $d = preg_replace('/\D/', '', $raw);
  $d = substr($d, -2);
  return str_pad($d, 2, '0', STR_PAD_LEFT);
}

/* ===== Entradas ===== */
$date2    = post_str('date2');              // YYYY-MM-DD (fecha del sorteo)
$loterias = (int)($_POST['loterias'] ?? 0); // idloteria
$number   = post_str('number');
$valor    = (float)($_POST['valor'] ?? 0);

/* ===== Validaciones ===== */
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date2)) {
  echo json_encode(['success'=>false,'error'=>'Fecha inválida (YYYY-MM-DD).']); exit;
}
if ($date2 < $fecha_hoy) {
  echo json_encode(['success'=>false,'error'=>'La fecha no puede ser menor a hoy.']); exit;
}
if ($loterias <= 0) {
  echo json_encode(['success'=>false,'error'=>'Seleccione una lotería.']); exit;
}
$number = to_2d_number($number);
if ($number === '' || !preg_match('/^\d{2}$/', $number)) {
  echo json_encode(['success'=>false,'error'=>'Número inválido (00..99).']); exit;
}
if ($valor < 0.50) {
  echo json_encode(['success'=>false,'error'=>'El valor debe ser ≥ 0.50.']); exit;
}

/* ===== Verificar ventana horaria de la lotería ===== */
$HCAST = "CAST(LPAD(LEFT(REPLACE(hora_fin, ':',''), 4), 4, '0') AS UNSIGNED)";
$sqlLot = "SELECT $HCAST AS finHHMM FROM tbr_loteria
           WHERE idloteria=? AND activar=1 AND condicion=1";
$st = $cn->prepare($sqlLot);
$st->bind_param('i', $loterias);
$st->execute();
$rs = $st->get_result();
if (!$row = $rs->fetch_assoc()) {
  $st->close();
  echo json_encode(['success'=>false,'error'=>'Lotería inactiva o inexistente.']); exit;
}
$st->close();
if ((int)$row['finHHMM'] < $hora_hoy_Hi && $date2 === $fecha_hoy) {
  echo json_encode(['success'=>false,'error'=>'Fuera de la hora de juego permitida.']); exit;
}

/* ===== Cupos ===== */
// cupo sucursal
$sql = "SELECT cupo FROM tbr_sucursal WHERE idsucursal=?";
$st = $cn->prepare($sql);
$st->bind_param('i', $sucursal_id);
$st->execute();
$rs = $st->get_result();
$cupo_maximo = (float)($rs->fetch_assoc()['cupo'] ?? 0);
$st->close();

// cupo extra del usuario (si tiene activo)
$sql = "SELECT extra_cupo FROM tbr_usuario WHERE idusuario=? AND cupo_activo=1";
$st = $cn->prepare($sql);
$st->bind_param('i', $id_usuario);
$st->execute();
$rs = $st->get_result();
$extra_cupo = (float)($rs->fetch_assoc()['extra_cupo'] ?? 0);
$st->close();

// suma ya jugada en sucursal para ese número/fdía (solo 2 cifras)
$sql = "SELECT IFNULL(SUM(t.valor),0) AS suma
        FROM tbr_ticket t
        INNER JOIN tbr_usuario u ON u.idusuario=t.id_usuario
        WHERE t.fecha=? AND t.condicion=1 AND t.id_cifras=1
          AND t.numero=? AND u.id_sucursal=?";
$st = $cn->prepare($sql);
$st->bind_param('ssi', $fecha_hoy, $number, $sucursal_id);
$st->execute();
$rs = $st->get_result();
$suma_valor = (float)($rs->fetch_assoc()['suma'] ?? 0);
$st->close();

$cupo_total     = $cupo_maximo + $extra_cupo;
$cupo_restante  = $cupo_total - $suma_valor;
if (($suma_valor + $valor) > $cupo_total) {
  echo json_encode([
    'success'        => false,
    'error'          => 'Cupo_no_disponible',
    'mensaje'        => "No puedes superar el cupo máximo de $cupo_total.",
    'cupo_restante'  => number_format(max($cupo_restante,0), 2, '.', '')
  ]);
  exit;
}

/* ===== Cálculo tope1/tope2 ===== */
$restante_sucursal = max($cupo_maximo - $suma_valor, 0);
$tope1 = min($valor, $restante_sucursal);
$tope2 = max($valor - $tope1, 0);
if ($tope2 > $extra_cupo) {
  echo json_encode([
    'success'=>false,
    'error'  => "No puedes superar el cupo adicional de $extra_cupo."
  ]); exit;
}

/* ===== Código aleatorio ===== */
function genCodigo(int $len=8): string {
  return substr(str_shuffle(str_repeat('0123456789', $len)), 0, $len);
}
$codigo   = genCodigo();
$estado   = 1;  // pendiente
$idCifras = 1;  // dos cifras
$cond     = 1;  // vigente

/* ===== Insert =====
 * tbr_ticket:
 *  fecha_impreso (DATE), hora_impreso (TIME),
 *  fecha (DATE), id_loteria (INT), numero (VARCHAR),
 *  valor (DECIMAL), id_usuario (INT), codigo (VARCHAR),
 *  estado (INT), tope1 (DECIMAL), tope2 (DECIMAL),
 *  id_cifras (INT), condicion (INT)
 */
$sql = "INSERT INTO tbr_ticket
  (fecha_impreso, hora_impreso, fecha, id_loteria, numero, valor,
   id_usuario, codigo, estado, tope1, tope2, id_cifras, condicion)
  VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?)";
$st = $cn->prepare($sql);
$types = "sssisdisiddii"; // 13 params
// Desglose: s s s i s d i s i d d i i
$st->bind_param(
  $types,
  $fecha_hoy,          // s
  $hora_hoy_hms,       // s
  $date2,              // s
  $loterias,           // i
  $number,             // s (varchar)
  $valor,              // d
  $id_usuario,         // i
  $codigo,             // s
  $estado,             // i
  $tope1,              // d
  $tope2,              // d
  $idCifras,           // i
  $cond                // i
);
$st->execute();
$idrifa = $cn->insert_id;
$st->close();

echo json_encode([
  'success' => true  
]);
