<?php
session_start();
include("conex.php");

error_reporting(E_ALL);
ini_set('display_errors', 1);

date_default_timezone_set('America/Guayaquil');

// =========================
//   SESIÓN / VARIABLES
// =========================
$idusuario  = isset($_SESSION['idusuario']) ? (int)$_SESSION['idusuario'] : 0;
$usuario    = htmlspecialchars($_SESSION['usuario'] ?? '');
$sucursal_s = htmlspecialchars($_SESSION['sucursal'] ?? '');
$rol        = isset($_SESSION['rol']) ? (int)$_SESSION['rol'] : 0;

// =========================
//   FILTROS RECIBIDOS
// =========================
$f_fecha_i   = $_POST['fecha_inicio'] ?? '';
$f_fecha_f   = $_POST['fecha_fin'] ?? '';
$id_loteria  = $_POST['id_loteria'] ?? '';  // filtro opcional por lotería

// Validación de fechas (formato YYYY-MM-DD)
if (!empty($f_fecha_i) && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $f_fecha_i)) {
    echo json_encode(["error" => "La fecha de inicio no es válida"], JSON_UNESCAPED_UNICODE);
    exit;
}
if (!empty($f_fecha_f) && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $f_fecha_f)) {
    echo json_encode(["error" => "La fecha de fin no es válida"], JSON_UNESCAPED_UNICODE);
    exit;
}
if (!empty($f_fecha_i) && !empty($f_fecha_f) && $f_fecha_f < $f_fecha_i) {
    echo json_encode(["error" => "La fecha de fin no puede ser anterior a la fecha de inicio"], JSON_UNESCAPED_UNICODE);
    exit;
}

// Validación de id_loteria (si viene)
if ($id_loteria !== '' && !ctype_digit((string)$id_loteria)) {
    echo json_encode(["error" => "El id_loteria no es válido"], JSON_UNESCAPED_UNICODE);
    exit;
}
$id_loteria = ($id_loteria !== '') ? (int)$id_loteria : 0;

// =========================
//   CONSULTA PRINCIPAL
// =========================
$sql = "SELECT 
            t.id_cifras,
            t.id_loteria,          -- ⭐ IMPORTANTE: id de la lotería
            l.loteria,
            c.cifras,
            t.fecha, 
            s.sucursal,
            t.id_usuario,
            u.usuario,
            SUM(t.valor) AS suma_valor
        FROM tbr_ticket AS t
        INNER JOIN tbr_cifras   AS c ON c.idcifras   = t.id_cifras
        INNER JOIN tbr_usuario  AS u ON u.idusuario  = t.id_usuario
        INNER JOIN tbr_sucursal AS s ON s.idsucursal = u.id_sucursal
        INNER JOIN tbr_loteria  AS l ON l.idloteria  = t.id_loteria
        WHERE t.condicion = 1 
          AND t.estado >= 2
          AND t.id_cifras = 2
          AND u.idusuario = ?";

$params = [];
$types  = "";

// idusuario siempre va
$params[] = $idusuario;
$types   .= "i";

// Filtro opcional por id_loteria (POST)
if ($id_loteria > 0) {
    $sql .= " AND t.id_loteria = ?";
    $params[] = $id_loteria;
    $types   .= "i";
}

// Filtro opcional por rango de fechas
if (!empty($f_fecha_i) && !empty($f_fecha_f)) {
    $sql .= " AND t.fecha BETWEEN ? AND ?";
    $params[] = $f_fecha_i;
    $params[] = $f_fecha_f;
    $types   .= "ss";
}

$sql .= " GROUP BY 
            t.id_cifras,
            t.id_loteria,          -- ⭐ también en el GROUP BY
            l.loteria, 
            t.fecha, 
            s.sucursal, 
            t.id_usuario, 
            u.usuario";

$stmt = mysqli_prepare($conn, $sql);
if ($stmt) {
    if (!empty($params)) {
        mysqli_stmt_bind_param($stmt, $types, ...$params);
    }
    mysqli_stmt_execute($stmt);
    $consulta = mysqli_stmt_get_result($stmt);
} else {
    error_log("Error en la preparación de la consulta: " . mysqli_error($conn));
    http_response_code(500);
    echo json_encode(["error" => "Error en la consulta"], JSON_UNESCAPED_UNICODE);
    exit;
}

// =========================
//   RECORRER RESULTADOS
// =========================
$tabla         = [];
$total_vendido = 0;
$total_premio  = 0;

while ($row = mysqli_fetch_assoc($consulta)) {
    $usuario_sql    = (int)$row['id_usuario'];
    $fecha_sql      = $row['fecha'];
    $idLoteriaFila  = (int)$row['id_loteria'];  // ⭐ lotería específica de esta fila

    // ==========
    //  PREMIOS
    // ==========
    $sql2 = "SELECT COALESCE(SUM(ts.premio), 0) AS Suma_premio
             FROM tbr_ticket_suerte AS ts
             INNER JOIN tbr_ticket  AS t ON t.idrifa = ts.id_ticket 
             INNER JOIN tbr_sorteo  AS s 
                 ON s.fecha_g    = t.fecha
                AND s.id_loteria = t.id_loteria  -- relación exacta sorteo–ticket
             WHERE ts.condicion = 1 
               AND t.condicion  = 1 
               AND t.estado     > 1
               AND ts.id_suerte = s.suerte_g 
               AND t.id_usuario = ?
               AND t.fecha      = ?
               AND t.id_cifras  = 2
               AND t.numero     = RIGHT(s.numero_g, 3)
               AND t.id_loteria = ?";          // ⭐ solo esta lotería

    $params2 = [];
    $types2  = "";

    // usuario, fecha y lotería de la fila
    $params2[] = $usuario_sql;
    $params2[] = $fecha_sql;
    $params2[] = $idLoteriaFila;
    $types2   .= "isi";

    $stmt2 = mysqli_prepare($conn, $sql2);
    if ($stmt2) {
        mysqli_stmt_bind_param($stmt2, $types2, ...$params2);
        mysqli_stmt_execute($stmt2);
        $consulta2 = mysqli_stmt_get_result($stmt2);

        $suma_p = 0;
        if ($row2 = mysqli_fetch_assoc($consulta2)) {
            $suma_p = (float)$row2['Suma_premio'];
        }

        mysqli_stmt_close($stmt2);
    } else {
        error_log("Error en la consulta de premios: " . mysqli_error($conn));
        $suma_p = 0;
    }

    // ==========
    //  CÁLCULOS
    // ==========
    $vendido = (float)$row['suma_valor'];
    $premio  = (float)$suma_p;
    $saldo   = $vendido - $premio;

    $total_vendido += $vendido;
    $total_premio  += $premio;

    $tabla[] = [
        "loteria"   => $row['loteria'],
        "cifras"    => $row['cifras'],
        "fecha"     => $row['fecha'],
        "sucursal"  => $row['sucursal'],
        "usuario"   => $row['usuario'],
        "vendido"   => number_format($vendido, 2, '.', ''),
        "premio"    => number_format($premio, 2, '.', ''),
        "saldo"     => number_format($saldo, 2, '.', '')
    ];
}

$total_utilidad = $total_vendido - $total_premio;

mysqli_close($conn);

// =========================
//   RESPUESTA JSON
// =========================
header('Content-Type: application/json; charset=utf-8');
echo json_encode([
    "data"    => $tabla,
    "totales" => [
        "total_vendido"  => number_format($total_vendido, 2, '.', ''),
        "total_premio"   => number_format($total_premio, 2, '.', ''),
        "total_utilidad" => number_format($total_utilidad, 2, '.', '')
    ]
], JSON_UNESCAPED_UNICODE);
