<?php
// funcion_r_up.php
session_start();
include("conex.php");

header('Content-Type: application/json; charset=utf-8');

// NO mostramos errores en pantalla para no romper el JSON
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

if (!isset($conn) || !$conn instanceof mysqli) {
    echo json_encode(["data" => [], "error" => "Sin conexión a la base de datos"], JSON_UNESCAPED_UNICODE);
    exit;
}

mysqli_set_charset($conn, 'utf8mb4');
date_default_timezone_set('America/Guayaquil');

// =========================
//   SESIÓN / VARIABLES
// =========================
$idusuario_sesion = isset($_SESSION['idusuario']) ? (int)$_SESSION['idusuario'] : 0;

// =========================
//   FILTROS RECIBIDOS
// =========================
// Llegan desde DataTables (POST JS)
$f_usuario   = isset($_POST['usuario'])      ? (int)$_POST['usuario']   : 0;
$f_sucursal  = isset($_POST['sucursal'])     ? (int)$_POST['sucursal']  : 0;
$f_fecha_i   = $_POST['fecha_inicio']        ?? '';
$f_fecha_f   = $_POST['fecha_fin']           ?? '';

// ❌ YA NO obligamos a usar el usuario de sesión
// if ($f_usuario <= 0 && $idusuario_sesion > 0) { $f_usuario = $idusuario_sesion; }

// =========================
//   VALIDACIÓN DE FECHAS
// =========================
if (!empty($f_fecha_i) && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $f_fecha_i)) {
    echo json_encode(["data" => [], "error" => "La fecha de inicio no es válida"], JSON_UNESCAPED_UNICODE);
    exit;
}
if (!empty($f_fecha_f) && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $f_fecha_f)) {
    echo json_encode(["data" => [], "error" => "La fecha de fin no es válida"], JSON_UNESCAPED_UNICODE);
    exit;
}
if (!empty($f_fecha_i) && !empty($f_fecha_f) && $f_fecha_f < $f_fecha_i) {
    echo json_encode(["data" => [], "error" => "La fecha de fin no puede ser anterior a la fecha de inicio"], JSON_UNESCAPED_UNICODE);
    exit;
}

// =========================
//   SQL PRINCIPAL (VENTAS)
// =========================
// 👉 IMPORTANTE: ahora también seleccionamos t.id_loteria para poder
// calcular los premios por lotería, y lo incluimos en el GROUP BY.
$sql = "
    SELECT 
        t.id_cifras,
        t.id_loteria,
        l.loteria,
        c.cifras,
        t.fecha, 
        s.sucursal,
        t.id_usuario,
        u.usuario,
        SUM(t.valor) AS suma_valor
    FROM tbr_ticket AS t
    INNER JOIN tbr_cifras   AS c ON c.idcifras   = t.id_cifras
    INNER JOIN tbr_usuario  AS u ON u.idusuario  = t.id_usuario
    INNER JOIN tbr_sucursal AS s ON s.idsucursal = u.id_sucursal
    INNER JOIN tbr_loteria  AS l ON l.idloteria  = t.id_loteria
    WHERE t.condicion = 1 
      AND t.estado    >= 2
      AND t.id_cifras = 1
";

$params = [];
$types  = "";

// Filtro por usuario SOLO si viene seleccionado
if ($f_usuario > 0) {
    $sql     .= " AND u.idusuario = ? ";
    $params[] = $f_usuario;
    $types   .= "i";
}

// Filtro por sucursal SOLO si viene seleccionada
if ($f_sucursal > 0) {
    $sql     .= " AND u.id_sucursal = ? ";
    $params[] = $f_sucursal;
    $types   .= "i";
}

// Filtro de fechas
if (!empty($f_fecha_i) && !empty($f_fecha_f)) {
    $sql     .= " AND t.fecha BETWEEN ? AND ? ";
    $params[] = $f_fecha_i;
    $params[] = $f_fecha_f;
    $types   .= "ss";
}

$sql .= "
    GROUP BY 
        t.id_cifras, 
        t.id_loteria,
        l.loteria, 
        t.fecha, 
        s.sucursal, 
        t.id_usuario, 
        u.usuario
    ORDER BY
        t.fecha ASC, l.loteria ASC
";

$stmt = mysqli_prepare($conn, $sql);
if (!$stmt) {
    error_log("Error en la preparación de la consulta principal: " . mysqli_error($conn));
    echo json_encode(["data" => [], "error" => "Error en la consulta principal"], JSON_UNESCAPED_UNICODE);
    exit;
}

if (!empty($params)) {
    mysqli_stmt_bind_param($stmt, $types, ...$params);
}
mysqli_stmt_execute($stmt);
$consulta = mysqli_stmt_get_result($stmt);

// =========================
//   CONSULTA DE PREMIOS
// =========================
// 👉 AHORA: premios por USUARIO + FECHA + LOTERÍA
$sql2 = "
    SELECT COALESCE(SUM(ts.premio), 0) AS Suma_premio
    FROM tbr_ticket_suerte AS ts
    INNER JOIN tbr_ticket  AS tt ON tt.idrifa = ts.id_ticket 
    INNER JOIN tbr_sorteo  AS so ON so.fecha_g = tt.fecha
                                  AND so.id_loteria = tt.id_loteria
    WHERE ts.condicion = 1 
      AND tt.condicion = 1 
      AND tt.estado    > 1
      AND ts.id_suerte = so.suerte_g 
      AND tt.id_usuario = ? 
      AND tt.fecha      = ?
      AND tt.id_loteria = ?
      AND (
           (tt.id_cifras = 1 AND (tt.numero = RIGHT(so.numero_g, 2)))
      )
";
$stmt2 = mysqli_prepare($conn, $sql2);
if (!$stmt2) {
    error_log("Error en la preparación de la consulta de premios: " . mysqli_error($conn));
}

// =========================
//   RECORRER RESULTADOS
// =========================
$tabla          = [];
$total_vendido  = 0.0;
$total_premio   = 0.0;

while ($row = mysqli_fetch_assoc($consulta)) {
    $usuario_sql   = (int)$row['id_usuario'];
    $fecha_sql     = $row['fecha'];
    $id_loteria_sql= (int)$row['id_loteria'];

    $suma_p = 0.0;

    if ($stmt2) {
        // 👇 ahora filtramos también por id_loteria
        mysqli_stmt_bind_param($stmt2, "isi", $usuario_sql, $fecha_sql, $id_loteria_sql);
        mysqli_stmt_execute($stmt2);
        $consulta2 = mysqli_stmt_get_result($stmt2);

        if ($row2 = mysqli_fetch_assoc($consulta2)) {
            $suma_p = (float)$row2['Suma_premio'];
        }
    }

    $vendido = (float)$row['suma_valor'];
    $premio  = $suma_p;
    $saldo   = $vendido - $premio;

    $total_vendido += $vendido;
    $total_premio  += $premio;

    $tabla[] = [
        "loteria"   => $row['loteria'],
        "cifras"    => $row['cifras'],
        "fecha"     => $row['fecha'],
        "sucursal"  => $row['sucursal'],
        "usuario"   => $row['usuario'],
        "vendido"   => number_format($vendido, 2, '.', ''),
        "premio"    => number_format($premio, 2, '.', ''),
        "saldo"     => number_format($saldo, 2, '.', '')
    ];
}

if ($stmt2) {
    mysqli_stmt_close($stmt2);
}
mysqli_stmt_close($stmt);
mysqli_close($conn);

$total_utilidad = $total_vendido - $total_premio;

echo json_encode([
    "data"    => $tabla,
    "totales" => [
        "total_vendido"  => number_format($total_vendido, 2, '.', ''),
        "total_premio"   => number_format($total_premio, 2, '.', ''),
        "total_utilidad" => number_format($total_utilidad, 2, '.', '')
    ]
], JSON_UNESCAPED_UNICODE);
