<?php
// datos.php — selects HTML de soporte (loterías, sucursal, cifras, usuarios, roles)
declare(strict_types=1);
header('Content-Type: text/html; charset=UTF-8');
date_default_timezone_set('America/Guayaquil');

/* ===== Helper de escape seguro (convierte a string) ===== */
function esc($v): string {
  return htmlspecialchars((string)$v, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

/* ===== Router ===== */
$accion = $_GET['Accion'] ?? null;
if ($accion) {
  switch ($accion) {
    case "GetLoterias":        GetLoterias();        break; // por fecha y dentro del rango horario actual (si es hoy)
    case "GetLoteriasP":       GetLoteriasP();       break; // por fecha (GET fechap)
    case "GetLoteriasorteo":   GetLoteriasorteo();   break; // por fecha (todas, sin filtrar por hora)
    case "GetSucursal":        GetSucursal();        break;
    case "GetCifras":          GetCifras();          break;
    case "GetUsuario":         GetUsuario();         break;
    case "GetRol":             GetRol();             break;
    case "GetLote":            GetLote();             break;
    default: echo "Acción no válida.";
  }
  exit;
}

/* ============================================================
 * Normaliza columna de hora a HHMM numérico desde TIME/VARCHAR:
 * - '06:00' -> 0600 -> 600 (al comparar lo tratamos como número)
 * - '06:00:00' -> 0600
 * - '600'/'0600' -> 0600
 * ============================================================ */
function hhmm_sql_expr(string $col): string {
  return "CAST(LPAD(LEFT(
            CASE
              WHEN LOCATE(':', $col) > 0 THEN REPLACE($col, ':','')
              WHEN $col REGEXP '^[0-9]+$' THEN $col
              ELSE '0000'
            END
          , 4), 4, '0') AS UNSIGNED)";
}

/* ============================================================
 * Loterías activas por fecha (sin validar hora)
 * POST: fecha=YYYY-MM-DD
 * ============================================================ */
function GetLoteriasorteo(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $fecha = $_POST['fecha'] ?? '';
  if (!$fecha) { echo "<option value=''>Fecha inválida</option>"; return; }

  $sql = "SELECT idloteria, loteria
          FROM tbr_loteria
          WHERE condicion = 1
            AND activar  = 1
            AND fecha    = ?
          ORDER BY hora_inicio ASC, loteria ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }

  mysqli_stmt_bind_param($stmt, "s", $fecha);
  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html = "<option value=''>Seleccione una lotería</option>";
  while ($row = mysqli_fetch_assoc($rs)) {
    $html .= "<option value='".esc($row["idloteria"] ?? '')."'>".esc($row["loteria"] ?? '')."</option>";
  }
  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}

/* ============================================================
 * Loterías activas por fecha y, si es HOY, dentro del rango horario actual
 * POST: fecha=YYYY-MM-DD, opcional _debug=1
 * ============================================================ */
function GetLoterias(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $fecha = $_POST['fecha'] ?? '';
  $debug = isset($_POST['_debug']) ? 1 : 0;
  if (!$fecha) { echo "<option value=''>Fecha inválida</option>"; return; }

  $esHoy = ($fecha === date('Y-m-d'));
  $hhmm  = (int)date('Hi'); // 06:05 -> 605, 14:35 -> 1435

  $HINI = hhmm_sql_expr('hora_inicio');
  $HFIN = hhmm_sql_expr('hora_fin');

  if ($esHoy) {
    // Respetar ventana horaria en la fecha actual
    $sql = "
      SELECT idloteria, loteria
      FROM tbr_loteria
      WHERE condicion = 1
        AND activar  = 1
        AND fecha    = ?
        AND $HINI <= ?
        AND $HFIN >= ?
      ORDER BY $HINI ASC, loteria ASC
    ";
    $stmt = mysqli_prepare($conn, $sql);
    if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }
    mysqli_stmt_bind_param($stmt, "sii", $fecha, $hhmm, $hhmm);
  } else {
    // Otras fechas: sin filtro de hora
    $sql = "
      SELECT idloteria, loteria
      FROM tbr_loteria
      WHERE condicion = 1
        AND activar  = 1
        AND fecha    = ?
      ORDER BY $HINI ASC, loteria ASC
    ";
    $stmt = mysqli_prepare($conn, $sql);
    if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }
    mysqli_stmt_bind_param($stmt, "s", $fecha);
  }

  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html  = "<option value=''>Seleccione una lotería</option>";
  $count = 0;
  while ($row = mysqli_fetch_assoc($rs)) {
    $count++;
    $html .= "<option value='".esc($row['idloteria'] ?? '')."'>".esc($row['loteria'] ?? '')."</option>";
  }

  if ($count === 0) {
    //if ($debug) {
      //html .= "<option value=''>[DEBUG] Sin resultados: fecha=".esc($fecha).", esHoy=".($esHoy?1:0).", hhmm={$hhmm}. "
       //      .  "Verifique activar=1, condicion=1 y horas (hora_inicio / hora_fin).</option>";
    //} else {
      $html .= "<option value=''>No hay loterías disponibles</option>";
    //}
  }

  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}

/* ============================================================
 * Loterías por fecha (GET fechap) – útil para filtros históricos
 * ============================================================ */
function GetLoteriasP(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $fecha = $_GET['fechap'] ?? '';
  if (!$fecha) { echo "<option value=''>Fecha inválida</option>"; return; }

  $sql = "SELECT idloteria, loteria
          FROM tbr_loteria
          WHERE condicion = 1
            AND fecha = ?
          ORDER BY hora_inicio ASC, loteria ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }

  mysqli_stmt_bind_param($stmt, "s", $fecha);
  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html = "<option value=''>Seleccione una lotería</option>";
  while ($row = mysqli_fetch_assoc($rs)) {
    $html .= "<option value='".esc($row["idloteria"] ?? '')."'>".esc($row["loteria"] ?? '')."</option>";
  }
  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}

/* ============================================================
 * Sucursales activas
 * ============================================================ */
function GetSucursal(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $sql = "SELECT idsucursal, sucursal
          FROM tbr_sucursal
          WHERE condicion = 1 AND activo = 1
          ORDER BY sucursal ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }

  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html = "<option value=''>Seleccione una sucursal</option>";
  while ($row = mysqli_fetch_assoc($rs)) {
    $html .= "<option value='".esc($row["idsucursal"] ?? '')."'>".esc($row["sucursal"] ?? '')."</option>";
  }
  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}

/* ============================================================
 * Cifras (2c, 3c, 4c, …)
 * ============================================================ */
function GetCifras(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $sql = "SELECT idcifras, cifras
          FROM tbr_cifras
          WHERE condicion = 1
          ORDER BY idcifras ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }

  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html = "<option value=''>Seleccione la cifra</option>";
  while ($row = mysqli_fetch_assoc($rs)) {
    $html .= "<option value='".esc($row["idcifras"] ?? '')."'>".esc($row["cifras"] ?? '')."</option>";
  }
  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}

/* ============================================================
 * Usuarios por sucursal
 * GET: Sucursal=<id>
 * ============================================================ */
function GetUsuario(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $sucursal = isset($_GET['Sucursal']) ? (int)$_GET['Sucursal'] : 0;

  $sql = "SELECT idusuario, usuario
          FROM tbr_usuario
          WHERE condicion = 1 AND activo = 1 AND id_sucursal = ?
          ORDER BY usuario ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }

  mysqli_stmt_bind_param($stmt, "i", $sucursal);
  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html = "<option value=''>Seleccione un usuario</option>";
  while ($row = mysqli_fetch_assoc($rs)) {
    $html .= "<option value='".esc($row["idusuario"] ?? '')."'>".esc($row["usuario"] ?? '')."</option>";
  }
  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}

/* ============================================================
 * Roles
 * ============================================================ */
function GetRol(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $sql = "SELECT idtipo_usuario, tipo
          FROM tbr_tipo_usuario
          WHERE condicion = 1
          ORDER BY tipo ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }

  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html = "<option value=''>Seleccione rol</option>";
  while ($row = mysqli_fetch_assoc($rs)) {
    $html .= "<option value='".esc($row["idtipo_usuario"] ?? '')."'>".esc($row["tipo"] ?? '')."</option>";
  }
  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}


/* ============================================================
 * seleccion GetLote
 * ============================================================ */
function GetLote(): void {
  include "conex.php";
  if (!isset($conn) || !$conn) { echo "<option value=''>Sin conexión</option>"; return; }

  $sql = "SELECT idpremio, detalle
          FROM tbr_premio
          WHERE condicion = 1
          ORDER BY idpremio ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { echo "<option value=''>Error de consulta</option>"; return; }

  mysqli_stmt_execute($stmt);
  $rs = mysqli_stmt_get_result($stmt);

  $html = "<option value=''>Seleccione loteria</option>";
  while ($row = mysqli_fetch_assoc($rs)) {
    $html .= "<option value='".esc($row["idpremio"] ?? '')."'>".esc($row["detalle"] ?? '')."</option>";
  }
  echo $html;

  mysqli_stmt_close($stmt);
  mysqli_close($conn);
}
